package de.das.encrypter.comm;

import java.net.DatagramPacket;
import java.net.DatagramSocket;

/**
 * The class provides a receiver service for UDP data packets on a specific port 
 * that continuously waits for incoming data and forwards it to the registered
 * data receiver. 
 * 
 * @author Frank Keldenich
 */
public class UdpReceiver extends Thread
{
  private DatagramSocket receiverSocket;
  
  private  DatagramPacket incommingPackage;
  
  private final int RECEIVER_BUFFER_SIZE = 4096 * 8;
  
  private final int udpReceiverPort;
  
  private DataReceiver dataReceiver;
  
  private boolean killed = false;
  
  private String receiverName = "";

  /**
   * Creates a new instance of the UDP receiver.
   * 
   * @param port the port that will be listened to.
   * @param recv the data expecting DataReceiver instance.
   * 
   * @throws Exception if the datagram socket couldn't be established.
   */
  public UdpReceiver(int port, DataReceiver recv) throws Exception
  {
    this ("", port, recv);
  }
  
  /**
   * Creates a new instance of the UDP receiver.
   * 
   * @param name a name for the receiver.
   * @param port the port that will be listened to.
   * @param recv the data expecting DataReceiver instance.
   * 
   * @throws Exception if the datagram socket couldn't be established.
   */
  public UdpReceiver(String name, int port, DataReceiver recv) throws Exception
  {
    receiverName = name;
    udpReceiverPort = port;
    dataReceiver = recv;
    init ();
  }
  
  private void init () throws Exception
  {
    receiverSocket = new DatagramSocket(udpReceiverPort);
    receiverSocket.setSoTimeout(0);
    receiverSocket.setReceiveBufferSize(RECEIVER_BUFFER_SIZE);
    incommingPackage = new DatagramPacket(
             new byte [RECEIVER_BUFFER_SIZE], RECEIVER_BUFFER_SIZE);
    setName ("UDP receiver service.");
    start();
  }

  /**
   * Set an instance which wants to get the received data.
   * @param dataReceiver 
   */
  public void setDataReceiver(DataReceiver dataReceiver) 
  {
    this.dataReceiver = dataReceiver;
  }
  
  /**
   * Heart of the thread. It runs in an endless loop and provides the service
   * continuously until the method "kill()" was invoked.
   * 
   * When data is received, it is forwarded to the registered data recipient 
   * along with the IP address of the sender.
   */
  @Override
  public void run ()
  {
    // Do it, until this thread dies.
    while (!killed)
    {
      // Wait for a connection and handle the incoming data.
      provideService();
    }
  }

  private void provideService() 
  {
    try
    {
      // Wait blocked for a client that wants a connection.
//System.out.println (receiverName + " waits blocked at port " + Integer.toString(udpReceiverPort));
      receiverSocket.receive(incommingPackage);
      if (!killed)
      {
//System.out.println (receiverName + " got data at port " + Integer.toString(udpReceiverPort));
        final byte [] data = new byte [incommingPackage.getLength()];
        System.arraycopy(incommingPackage.getData(), 0, data, 0, data.length);
        Thread th = new Thread ()
        {
          @Override
          public void run()
          {
            dataReceiver.setData(incommingPackage.getAddress().getHostAddress(), data);
//System.out.println (receiverName + " request processed.");
          }
        };
        th.start();
      }
    }
    catch (Exception iox)
    {
//      iox.printStackTrace();
    }
  }

  /**
   * Sets the killed-flag, closes the socket and performs an interrupt.
   */
  public void kill()
  {
    killed = true;
    receiverSocket.close();
    interrupt();
  }

}
