package de.das.encrypter.comm;

import java.net.DatagramPacket;
import java.net.DatagramSocket;

import java.net.InetAddress;
import java.net.SocketException;

/**
 * A simple UDP sender expecting a receiver service for a given host address at
 * the given port, which is able to receive and interpret the data sent.
 * 
 * @author KeldenichF
 */
public class UdpSender 
{
  private DatagramSocket senderSocket = null;
  
  private final int udpReceiverPort;
  
  private final String host;
  
  private String senderName = "";

  /**
   * Creates a new sender client instance and provides a datagram socket for
   * connection and sending.
   * 
   * @param port the port number where a service is expected at the host.
   * 
   * @throws Exception if the datagram socket cannot be establisched.
   */
  public UdpSender(int port) throws Exception 
  {
    this("127.0.0.1", port);
  }

  /**
   * Creates a new sender client instance and provides a datagram socket for
   * connection and sending.
   * 
   * @param host a predefined host address for later usage.
   * @param port the port number where a service is expected at the host.
   * 
   * @throws Exception if the datagram socket cannot be establisched.
   */
  public UdpSender(String host, int port) throws Exception 
  {
    this ("Unnamed", host, port);
  }
  
  /**
   * Creates a new sender client instance and provides a datagram socket for
   * connection and sending.
   * 
   * @param name of the sender.
   * @param host a predefined host address for later usage.
   * @param port the port number where a service is expected at the host.
   * 
   * @throws SocketException if the datagram socket cannot be establisched.
   */
  public UdpSender(String name, String host, int port) throws SocketException 
  {
    senderName = name;
    this.host = host;
    udpReceiverPort = port;
    senderSocket = new DatagramSocket();
  }
  
  /**
   * Tries to send the given byte array as a UDP message to the port of the 
   * predefined host.<br>
   * @param data the byte array to be sent.
   */
  public synchronized void callServer (byte [] data)
  {
    callServer (host, data);
  }
  
  /**
   * Tries to send the given byte array as a UDP message to the port of the 
   * given host.<br>
   * @param host the address of the receiver.
   * @param data the byte array to be sent.
   */
  public synchronized void callServer (String host, byte [] data)
  {
    try
    {
      DatagramPacket dpn = new DatagramPacket (
              data, data.length, 
              InetAddress.getByName(host), 
              udpReceiverPort);
      senderSocket.send(dpn);
//System.out.println (senderName + " sent to " + Integer.toString(udpReceiverPort));
    }
    catch (Exception ex)
    {
      ex.printStackTrace();
      // Currently not handled
    }
  }
}
