package de.das.encrypter.cryptoservice;

import de.das.encrypter.comm.DataReceiver;
import de.das.encrypter.comm.UdpReceiver;

import de.das.encrypter.model.EntryPointSupplier;
import de.das.encrypter.model.KeyFiles;

import de.das.encrypter.processors.BlockSender;
import de.das.encrypter.processors.EncoderDecoder;
import de.das.encrypter.processors.ResultReceiver;
import de.das.encrypter.processors.TransferControl;

import de.das.encrypter.tools.HexTool;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.util.StringTokenizer;

import javax.swing.Timer;

/**
 * Keys can be requested via this server. It receives requests consisting of an 
 * encrypted string containing a key ID and the desired length of the key 
 * separated by a semicolon. If a key with this ID is available and it is 
 * possible to provide the required length, the server starts sending a key 
 * block by block.
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class CryptoServer implements DataReceiver, ActionListener, ResultReceiver
{
  public static final String RESPONSE_ID = "CryptoServiceResponse";
  
  public static final int RESPONSE_ID_LENGTH = RESPONSE_ID.length();
  
  public static final int NO_REJECTION = 0;
  
  public static final int KEY_UNKNOWN = 1;
  
  public static final int KEYS_TOO_SHORT = 2;
  
  public static final int NO_UNUSED_KEY_AVAILABLE = 3;
  
  private static Long cryptoKeyId = null;
  
  private UdpReceiver recv;
  
  private static CryptoServer cs;
  
  private final KeyFiles kfs;
  
  private final EncoderDecoder ed = new EncoderDecoder();

  private final int CONNECTION_TIME_OUT_TIME = 2000;
  
  private BlockSender blockSender = null;
  
  private Timer connectionTimeOut;
  
  private TransferControl transferControl;
  
  private int port;
  
  private final EntryPointSupplier eps;
  
  private final int MAX_REQUEST_LENGTH = 80;
  
  private int rejectionReason = NO_REJECTION;
  
  public static void createInstance (KeyFiles kfs, int port, EntryPointSupplier eps) throws Exception
  {
    cs = new CryptoServer(kfs, port, eps);
  }

  public CryptoServer(KeyFiles kfs, int port, EntryPointSupplier eps) throws Exception
  {
    this.kfs = kfs;
    this.port = port;
    this.eps = eps;
    
    init ();
  }

  /**
   * Sets the ID of the key bundle to be used to encrypt the key part to be 
   * returned based on the request.
   * 
   * @param id the ID of a key bundle.
   */
  public static void setCryptoKeyId(Long id)
  {
    cryptoKeyId = id;
  }

  /**
   * Returns the current ID of the key used for encryption of requests and keys.
   * @return 
   */
  public static Long getCryptoKeyId() 
  {
    return cryptoKeyId;
  }
  
  /**
   * Changes the port where the crypto service is offered.
   * @param port the new port number.
   */
  public static void newPort (int port)
  {
    cs.changePort(port);
  }
  
  private void changePort (int port)
  {
    this.port = port;
    initCommunication();
  }
  
  private void init () throws Exception
  {
    EncoderDecoder.setup("", kfs);
    transferControl = new TransferControl(this);
    transferControl.setupSender(null, port);
    connectionTimeOut = new Timer (CONNECTION_TIME_OUT_TIME, this);
    initCommunication();
  }
  
  /**
   * Unused interface methode.
   * @param data 
   */
  @Override
  public void setData(byte[] data) 
  {
  }

  /**
   * Receives data and checks if the data length is smaller than 80 bytes, the 
   * data is encrypted, if the required key for decryption is available and if a 
   * "crypto key ID" has been set for the encryption of the key to be sent. If these 
   * conditions are met, the received data will be passed for decryption and if 
   * executable to start the broadcast. If one of these conditions is not met, 
   * the request is ignored.
   * 
   * @param host the IP-address of the receiver.
   * @param data received data via UDP service receiver.
   */
  @Override
  public void setData(String host, byte[] data) 
  {
    try 
    {
      transferControl.setHost (host, port);
    } 
    catch (Exception e) 
    {
      // Should not occur but ignore it if nevertheless.
    }
    rejectionReason = NO_REJECTION;
    if (data.length < MAX_REQUEST_LENGTH && EncoderDecoder.isEncrypted(data))
    {
      try 
      {
        long id = EncoderDecoder.getKeyId(data);
        if (kfs.containsKey(id) && cryptoKeyId != null && kfs.containsKey(cryptoKeyId))
        {
          parseRequestAndStartTransfer (host, data);
        }
      } 
      catch (Exception e) 
      {
        // Ignore the message if parsing for the key ID fails.
      }
      if (rejectionReason != NO_REJECTION)
      {
        byte [] res = new byte [RESPONSE_ID_LENGTH + 1];
        System.arraycopy(RESPONSE_ID.getBytes(), 0, res, 0, RESPONSE_ID_LENGTH);
        res[RESPONSE_ID_LENGTH] = (byte)rejectionReason;
        transferControl.getSender().callServer(res);
      }
    }
  }

  private void parseRequestAndStartTransfer(String host, byte[] data) throws Exception
  {
    data = ed.decrypt(data);
    String request = new String (data);
    if (request.contains(";"))
    {
      StringTokenizer tokens = new StringTokenizer(request, ";");
      if (tokens.countTokens() == 2)
      {
        String keyBundleName = tokens.nextToken();
        int size = Integer.parseInt(tokens.nextToken());
        Long id = kfs.getKeyIdByKeyName(keyBundleName);
        if (id != null)
        {
          byte [] pureKey = kfs.get(id).getPureKeyData();
          if (pureKey.length >= size)
          {
            startTransfer (host, id, pureKey, size);
          }
          else
          {
            rejectionReason = KEYS_TOO_SHORT;
          }
        }
        else
        {
          rejectionReason = KEY_UNKNOWN;
        }
      }
    }
  }

  private void startTransfer(String host, long id, byte [] key, int size) throws Exception
  {
    transferControl.setHost(host, port);
    int KEY_ID_LENGTH = 8;
    int ENTRY_POINT_LENGTH = 4;
    int HEADER_LENGTH = KEY_ID_LENGTH + ENTRY_POINT_LENGTH;
    byte [] keyData = new byte [size + HEADER_LENGTH];
    Integer entryPoint = eps.getEntryPoint(id, key);
    if (entryPoint != null)
    {
      HexTool.insertLongLittleEndianAt(id, 0, keyData);
      HexTool.insertIntLittleEndianAt(entryPoint, KEY_ID_LENGTH, keyData);
      for (int i = 0; i < size; i++)
      {
        keyData[i + HEADER_LENGTH] = key[entryPoint++];
        entryPoint = entryPoint == key.length ? 0 : entryPoint;
      }
      keyData = ed.encrypt(entryPoint, kfs.get(cryptoKeyId).getKey(), keyData, "---");

      recv.setDataReceiver(transferControl);
      transferControl.setBreakSender(false);
      transferControl.setTransferResult(TransferControl.NO_RESULT);
      connectionTimeOut.start();
      blockSender = new BlockSender(keyData, null, transferControl, connectionTimeOut);
      blockSender.start();

      while (transferControl.getTransferResult() == TransferControl.NO_RESULT && 
            !transferControl.isBreakSender())
      {
        try 
        {
          Thread.sleep(50);
        } 
        catch (Exception e) 
        {

        }
      }
      recv.setDataReceiver(this);
    }
    else
    {
      rejectionReason = NO_UNUSED_KEY_AVAILABLE;
    }
  }

  private void initCommunication()
  {
    if (recv != null)
    {
      recv.kill();
    }
    try
    {
      recv = new UdpReceiver("Crypto Server", port + 1, transferControl);
      recv.setDataReceiver(this);
    } 
    catch (Exception e)
    {
      e.printStackTrace();
    }
  }
  
  /**
   * Receiver of an action event in case of a time out situation.
   * 
   * @param e an action event.
   */
  @Override
  public void actionPerformed(ActionEvent e)
  {
    if (e.getSource().equals(connectionTimeOut))
    {
      connectionTimeOut.stop();
      transferControl.setBreakSender(true);
      if (blockSender != null || transferControl.isWaitingForInfo())
      {
        if (blockSender != null)
        {
          blockSender.interrupt();
          blockSender = null;
        }
        initCommunication();
      }
    }
  }

  /**
   * The result information coming from the transfer control instance.
   * 
   * @param result an integer with information whether a transfer was successful 
   * or failed.
   * @param info an optional and additional information text.
   */
  @Override
  public void setResult (int result, String info)
  {
    switch (result)
    {
      case ResultReceiver.PARTNER_RESPONSE:
        connectionTimeOut.stop();
      break;
      case ResultReceiver.TRANSFER_FAILED:
        connectionTimeOut.stop();
        transferControl.setTransferResult (TransferControl.FAILURE);
      break;
      case ResultReceiver.TRANSFER_VALID:
        connectionTimeOut.stop();
        transferControl.setTransferResult (TransferControl.SUCCESS);
      break;
      case ResultReceiver.TRANSFER_BREAK:
        connectionTimeOut.stop();
      break;
    }
  }
  
}
