package de.das.encrypter.examples;

import de.das.encrypter.comm.DataReceiver;
import de.das.encrypter.comm.UdpReceiver;
import de.das.encrypter.comm.UdpSender;
import de.das.encrypter.cryptoservice.CryptoServer;

import de.das.encrypter.model.KeyFile;
import de.das.encrypter.model.KeyFiles;

import de.das.encrypter.processors.Assembler;
import de.das.encrypter.processors.EncoderDecoder;
import de.das.encrypter.processors.KeyTransfer;
import de.das.encrypter.processors.TransferControl;

import de.das.encrypter.tools.CRC;
import de.das.encrypter.tools.HexTool;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.net.SocketException;

import java.util.Arrays;

import javax.swing.Timer;

/**
 * This example requests a key from the Encrypter-X crypto service, encrypts an 
 * image, and writes it back encrypted to a file. The requested and used key is 
 * in memory only for a short time and only during use. After running this 
 * example, the encrypted file can be viewed using the Encrypter-X browser 
 * function.<br><br>
 * The requirements for running this example are that the Encrypter-X is running 
 * on this or another machine, its crypto service is configured (port is 
 * defined), the Encrypter-X and this example know the same keyring, and this 
 * example knows the IP address of the crypto service's home (usually 
 * localhost="127.0.0.1").
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich (2023-03-18)
 */
public class CryptoServiceUser implements ActionListener, DataReceiver
{
  // Path to the image that shall be encrypted.
  private final String IMAGE_FILE_PATH = ".\\Tests\\meer50.jpg";
  // Path to the file that shall contain the encrypted image.
  private final String ENCRYPTED_IMAGE_FILE_PATH = ".\\Tests\\af45e90d3210fa3";
  
  // Enter the path to an existing file with a key bundle.
  private final String KEY_PATH = ".\\Tests\\128kB";
  
  // A key name that must be known to the Encrypter-X.
  private final String KEY_NAME = "5MB";
  
  // The instance of the encoder decoder, the class that performs encryption
  // and decryption.
  private final EncoderDecoder ed = new EncoderDecoder();
  
  // The UDP receiver with which the crypto service can communicate.
  private UdpReceiver recv;
  
  // The UDP sender with which the crypto service can be addressed.
  private UdpSender sender;
  
  // The default port of the crypto service. If it is configured differently, 
  // it must be changed here accordingly.
  private final int port = 44444;
  
  // IP address of the computer where the Crypto service is located.
  private String host = "127.0.0.1";
  
  // The class that will collect the incoming blocks of the key transfer.
  // It is instantiated when the first block is received and with it the 
  // information about the length of the key to be sent.
  private Assembler assembler = null;
  
  // The buffer that holds the received blocks.
  private byte[] buffer;
  
  // A variable that counts which block is currently expected.
  private int expectedBlockCount;
  
  private final int CONNECTION_TIME_OUT_TIME = 2000;
  
  // Timer to monitor the transfer and break it, if incoming blocks are missing.
  private Timer connectionTimeOut;
  
  // The map that hold the crypto key for encrypting the request and decrypting
  // the received key.
  private KeyFiles kfs;
  
  // A flag that indicates the end of the transfer.
  private static boolean received = false;
  
  // A flag indicating successful reception.
  private static boolean success;
  
  // A flag indicating that a timeout has occurred.
  private static boolean timeout;
  
  // The information of the crypto service why the request was rejected.
  private static int rejectReason;
  
  public static void main (String [] args)
  {
    try 
    {
      new CryptoServiceUser().execute();
      // Request has been sent. Now wait for the crypto service response.
      while (!received)
      {
        Thread.sleep(100);
      }
      // Anything has terminated the request. Printout the reason.
      if (success)
      {
        // Everything ok. The key was received, the image was encrypted and 
        // saved.
        System.out.println ("Request successful");
      }
      else
      {
        if (timeout)
        {
          // A timeout has ended the request.
          System.out.println ("Crypto Service not available.");
        }
        else
        {
          // The server responded but rejected the request and gave the reason.
          switch (rejectReason)
          {
            case CryptoServer.KEY_UNKNOWN:
              System.out.println ("Rejection because unknown key bundle.");
            break;
            case CryptoServer.KEYS_TOO_SHORT:
              System.out.println ("Rejection because requested key too long.");
            break;
            case CryptoServer.NO_UNUSED_KEY_AVAILABLE:
              System.out.println ("Rejection because no more unused key available.");
            break;
            default:
              System.out.println ("Rejection because unknown reason.");
          }
        }
      }
    } 
    catch (Exception e) 
    {
      // Can occur if the receiver port is in use and not available.
      e.printStackTrace();
    }
    System.exit(0);
  }

  /**
   * Create a new instance of CryptoServiceUser.
   * 
   * @throws Exception if the receiver port is not available.
   */
  public CryptoServiceUser() throws Exception 
  {
    init();
  }

  /**
   * Initializes the communication.
   * 
   * @throws Exception if the receiver port is not available.
   */
  private void init() throws Exception
  {
    // Initialize the timeout timer.
    connectionTimeOut = new Timer (CONNECTION_TIME_OUT_TIME, this);
    // Set up receiver service and transmitter.
    prepareReceiver();
    prepareSender();
  }
  
  /**
   * If the crypto service is not available or our request could not be 
   * fulfilled, the program will timeout.
   * 
   * @param e an action event.
   */
  @Override
  public void actionPerformed(ActionEvent e)
  {
    if (e.getSource().equals(connectionTimeOut))
    {
      // Stopping the timer.
      connectionTimeOut.stop();
      // Sets the flags that indicate the reason for application termination.
      received = true;
      timeout = true;
    }
  }
  
  /**
   * Execution of the application.
   * 
   * @throws Exception in case of any unexpected problem.
   */
  private void execute() throws Exception
  {
    // Preset timeout with false.
    timeout = false;
    // Create an instance of the image file that shall be encrypted to get its 
    // length.
    File imageFile = new File (IMAGE_FILE_PATH);
    // Get the key bundle from a file to be used for the encryption of the
    // request.
    KeyFile kf = KeyFile.takeAsKey(new File (KEY_PATH));
    
    // The specified file was accepted as a bundle of keys.
    if (kf != null)
    {
      // Create the map for managing keys that is required by the EncoderDecoder.
      kfs = new KeyFiles();
      // Put the key bundle into the map.
      kfs.put (kf.getId(), kf);
      
      // Setup the encoder decoder by making the common key bundle public to it.
      EncoderDecoder.setup("", kfs);
      
      // Create the request to send a key. This assumes that the name of the 
      // file is known, which contains a keyring on the server side with keys 
      // long enough to encrypt the image data.
      String request = KEY_NAME + ";" + (int)imageFile.length();
      // Encrypt it with an arbitrary entry point e.g. 400.
      byte [] requestData = ed.encrypt(400, kf.getKey(), request.getBytes());
      // Start the timeout monitor.
      connectionTimeOut.start();
      // Send the request to the crypto service.
      sender.callServer("127.0.0.1", requestData);
    }
  }

  /**
   * Incoming data from the crypto service.
   * 
   * @param host the IP-address of the sender.
   * @param data the byte array sent by the sender.
   */
  @Override
  public void setData(String host, byte[] data)
  {
    setData(data);
  }

  /**
   * Incoming data from the crypto service. There can be three different types 
   * of messages. It can be a block of the requested data, where the first block 
   * initializes the reception, it can be an information why the request was 
   * rejected or it is the message that the transmission is finished.
   * 
   * @param data the byte array sent by the sender.
   */
  @Override
  public void setData(byte[] data)
  {
    // Stop timeout monitoring.
    connectionTimeOut.stop();
    
    // Check if it is the EOT information.
    if (data.length == KeyTransfer.EOT.length &&
        Arrays.equals(data, KeyTransfer.EOT))
    {
      // Transfer ended without any error.
      success = true;
      // So go and get the received key from the assembler.
      saveKey();
    }
    // Check if it is a rejection information.
    else if (data.length > CryptoServer.RESPONSE_ID_LENGTH && 
        HexTool.byteToString(
          data, 0, CryptoServer.RESPONSE_ID_LENGTH).equals(
                   CryptoServer.RESPONSE_ID))
    {
      // Server responses but due to any reason it could not perform the request.
      success = false;
      // Get the reason ID from the received data.
      rejectReason = (int)data[CryptoServer.RESPONSE_ID_LENGTH];
      // Set the received flag for the application can print out the reason and 
      // terminate.
      received = true;
    }
    // Now it can be only one block of the transmission.
    else 
    {
      // If this is the first block, the assembler does not yet exist and must 
      // be created.
      if (assembler == null)
      {
        // First block seems to be received. Initialize the block assembler
        try 
        {
          // Data is the first received block. Therefore, total length of the 
          // transmission, which together with the block counter is always 
          // prefixed as header, is taken for initialization of the assembler.
          int dataSize = HexTool.byteToIntBigEndian(
                  KeyTransfer.POS_KEY_LENGTH, 
                  data);
          
          // Start the timeout timer again since we expect the next block or the 
          // EOT (end of transmission) information.
          connectionTimeOut.start();
          // Create an assembler and tell it the expected data size.
          assembler = new Assembler(dataSize, null);
          // Create a buffer with the block size to receive the following blocks.
          buffer = new byte [KeyTransfer.BUFFER_SIZE];
          // Preset the expected block count to "0".
          expectedBlockCount = 0;
          // Preset the success flag with false. It will be set to true not 
          // before the service reports EOT.
          success = false;
        } 
        catch (Exception e) 
        {
          // Inform the service that the transfer was aborted due to an error.
          sender.callServer((TransferControl.TRANSFER_BREAK).getBytes());
        }
      }
      else
      {
        // Restart the timeout timmer.
        connectionTimeOut.restart();
      }
      
      // Get the size of valid data coming with this input.
      int bufferSize = HexTool.byteToIntBigEndian(KeyTransfer.POS_DATA_SIZE, data);
      // If the amount of incoming data is smaller than the default block size
      // it is the last block with smaller size. Therefor decrease the buffer.
      if (bufferSize != buffer.length)
      {
        buffer = new byte [bufferSize];
      }
      // Copy the pure data (without header info) into the buffer.
      System.arraycopy(data, KeyTransfer.POS_BLOCK_START, buffer, 0, buffer.length);
      // Get the current block count from the header.
      int currentBlockCount = HexTool.byteToIntBigEndian(
              KeyTransfer.POS_BLOCK_COUNT, 
              data);
      // Get the CRC value from the end of the incoming data.
      int crc = HexTool.byteToIntBigEndian(data.length - 4, data);
      // Build the CRC value from the incoming data without the last four byte
      // and compare it with the value from the block and compare the current
      // block number with the expected.
      if (crc == (int)CRC.getCRC_32(data, 4) && 
          currentBlockCount == expectedBlockCount)
      {
        // Append the received block to the already received data.
        assembler.setData(buffer);
        // Increment the expected block counter.
        expectedBlockCount++;
        // Send it to the service for it can send the next block.
        sender.callServer(HexTool.intToByteArray(currentBlockCount));
      }
      else
      {
        // Anything wrong with the current block.
        // Inform the service that the transfer was aborted due to an error.
        sender.callServer((TransferControl.TRANSFER_BREAK).getBytes());
      }
    }
  }
  
  /**
   * Creates a UDP sender to request a key and acknowledge the incoming blocks.
   * 
   * @throws SocketException if the socket could not be set up on the requested 
   * port.
   */
  private void prepareSender() throws SocketException
  {
    // Create a UDP sender.
    sender = new UdpSender("CSU Sender", host, port + 1);
  }
  
  /**
   * Creates a UDP receiver service to receive the incoming blocks.
   * @throws Exception  if the receiver port is not available.
   */
  private void prepareReceiver() throws Exception
  {
    // Terminates the UDP receiver first if it already exists so that the socket 
    // is closed.
    if (recv != null)
    {
      recv.kill();
    }
    // Creates a new UDP receiver and makes itself known as a data receiver.
    recv = new UdpReceiver("USC Receiver", port, this);
  }
  
  /**
   * Takes the received key decrypts it and stores the image.
   */
  private void saveKey()
  {
    // Tell the crypto service that we got the transmission successfully.
    sender.callServer(KeyTransfer.VALID);
    // Get the assembled data.
    byte [] data = assembler.getData();
    try 
    {
      // Decrypt the received data.
      data = ed.decrypt(data);
    } 
    catch (Exception e) 
    {
      // If something goes wrong during decryption.
      e.printStackTrace();
    }
    // The actual key is preceded by the key ID (8 byte) and the entry point 
    // (4 byte).
    int KEY_ID_LENGTH = 8;
    int ENTRY_POINT_LENGTH = 4;
    int HEADER_LENGTH = KEY_ID_LENGTH + ENTRY_POINT_LENGTH;
    // Get the key ID from the header of the received and decrypted data.
    long id = HexTool.byteToLongLittleEndian(data, 0);
    // Get the entry point that has been used to create the key data from the header.
    int entryPoint = HexTool.byteToIntLittleEndian(KEY_ID_LENGTH, data);
    // Create a byte array to contain the pure key data without the header.
    byte [] pureKey = new byte [data.length - HEADER_LENGTH];
    // Copy the pure key data.
    System.arraycopy(data, HEADER_LENGTH, pureKey, 0, pureKey.length);
    // Just for information.
    System.out.println ("Key id: " + id);
    System.out.println ("Entry point: " + entryPoint + " Length: " + (data.length - 4));
    assembler = null;
    try 
    {
      File imageFile = new File (IMAGE_FILE_PATH);
      // Read the byte of an jpg image into a buffer.
      byte [] imgBuffer = new byte [(int)imageFile.length()];
      try (BufferedInputStream in = 
              new BufferedInputStream(
                      new FileInputStream(imageFile)))
      {
        in.read(imgBuffer);
      }
      
      // Encryption of the image data.
      byte [] eData = ed.encrypt(id, pureKey, entryPoint, imgBuffer, imageFile.getName());
      
      // Write encrypted data into a new file.
      try (BufferedOutputStream out = 
              new BufferedOutputStream(
                      new FileOutputStream(ENCRYPTED_IMAGE_FILE_PATH)))
      {
        out.write(eData);
      }
    } 
    catch (Exception e) 
    {
      // In case of any issue with image file reading, encryption or encrypted 
      // data writing.
      e.printStackTrace();
    }
    // Set the flag to true for this program can be terminated.
    received = true;
  }
}
