package de.das.encrypter.examples;

import de.das.encrypter.model.KeyFile;
import de.das.encrypter.model.KeyFiles;

import de.das.encrypter.processors.EncoderDecoder;

import de.das.encrypter.tools.HexTool;

import java.io.File;

/**
 * This example shows how data can be encrypted and decrypted. This assumes that 
 * one or more suitable key bundles exist or are generated beforehand. It shows 
 * the necessary steps to initialize the encryption factory and to allow 
 * encryption and decryption.
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class EncryptionDecryptionExample 
{
  // Enter the path to an existing file with a key.
  private final String KEY_PATH = "C:\\Bundle\\1MB";
  
  // A sentence to demonstrate the encryption
  private final String ORIGINAL_TEXT = "This example shows how data can be encrypted and decrypted.";
  
  public static void main (String [] args)
  {
    try 
    {
      new EncryptionDecryptionExample().execute();
    } 
    catch (Exception e) 
    {
      e.printStackTrace();
    }
    System.exit(0);
  }

  private void execute() throws Exception
  {
    // Get the key bundle from a file to be used for the encryption and
    // decryption.
    KeyFile kf = KeyFile.takeAsKey(new File (KEY_PATH));
    
    // The specified file was accepted as a bundle of keys.
    if (kf != null)
    {
      // Create map for managing key bundles.
      KeyFiles kfs = new KeyFiles();
      // Places the key bundle in the list.
      kfs.put (kf.getId(), kf);
      
      // Set up the encoder-decoder by telling it the key bundle list.
      EncoderDecoder.setup(kfs);
      // Create an instance of the encoder decoder.
      EncoderDecoder ed = new EncoderDecoder();
      // Specify an arbitrary entry point e.g. 126.
      int entryPoint = 126;
      // Get the part of the key that will be used for encryption.
      // This is only for the output of this demonstration.
      byte [] usedKeyPart = KeyFile.provideKeyPart(
              kf.getKey(), 
              entryPoint, 
              ORIGINAL_TEXT.length());
      // Now encrypt the original text.
      byte [] encryptedData = ed.encrypt(
              entryPoint, 
              kf.getKey(), 
              ORIGINAL_TEXT.getBytes());
      // Print it as ASCII text..
      System.out.println (ORIGINAL_TEXT);
      // Print the bytes of the ASCII text.
      System.out.println ("\nOriginal text bytes:");
      HexTool.printlnBuffer(ORIGINAL_TEXT.getBytes());
      // Print the bytes of the part of the key used for encryption.
      System.out.println ("\nKey part bytes:");
      HexTool.printlnBuffer(usedKeyPart);
      // Print the encrypted data.
      System.out.println ("\nEncrypted data bytes:");
      HexTool.printlnBuffer(ed.getPureEncryptedData(encryptedData));
      // Print the encrypted data together with the encryption header.
      System.out.println ("\nEncrypted data bytes plus header:");
      HexTool.printlnBuffer(encryptedData);
      // Print the bytes of the decrypted data. The key is for decryption will
      // be found by the decryption method automatically since it is contained
      // in the key files map that is known by the encoder decoder due to the 
      // setup above.
      byte [] decryptedData = ed.decrypt(encryptedData);
      System.out.println ("\nDecrypted data bytes:");
      // Print the decrypted bytes converted to a string.
      HexTool.printlnBuffer(decryptedData);
      System.out.println (new String (decryptedData));
    }
  }
}
