package de.das.encrypter.examples;

import de.das.encrypter.model.KeyFile;

import de.das.encrypter.tools.HexTool;
import de.das.encrypter.tools.KeyFileFileFilter;
import de.das.encrypter.tools.KeyQualityAssessor;
import de.das.encrypter.tools.Tools;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.awt.event.WindowEvent;

import java.awt.image.BufferedImage;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.InputStream;

import java.net.URL;

import javax.imageio.ImageIO;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * An application that analyzes the quality of key bundles. It examines the 
 * uniform distribution of byte values and the size of the scattering (deviation) 
 * between the uniform distribution values.
 * 
 * The uniform distribution of the frequency of occurrence of the byte values 
 * between 1 and 255 (normalized to the maximum value) is visualized, whereby 
 * all values should be at 1 if possible for high quality.
 * 
 * The path to the key bundle file to be analysed must be in the first line of
 * a text file named "Probe" in the relative folder "./Test".
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich (2023-03-18)
 */
public class KeyAnalyzer extends JFrame implements ActionListener
{
  private final Dimension SIZE = new Dimension (440, 440);
  
  private final Dimension BUTTON_SIZE = new Dimension (32, 32);
  
  private final String SOURCE_FILE_NAME_CONTAINER = ".\\Tests\\Probe";
  // Use a path to a key bundle file, get it from file selection.
  private String source = null;

  private final String LAST_PATH = "./PATH";
  
  private String lastSelectedPath;
  
  // Create an array of 256 float values for the analysis result that will be 
  // displayed.
  private final float [] norm = new float [256];
  
  // Create an array to hold the calculated quality values to be used by the 
  // display.
  private Double [] qualityValues;
  
  private JButton btnOpen;
  
  private File bundleFile;
  
  private Image img;
  
  private byte [] key;

  private Display display;
  
  public static void main (String [] args)
  {
    try
    {
      new KeyAnalyzer();
    } 
    catch (Exception e)
    {
      e.printStackTrace();
    }
  }

  /**
   * Create a new instance of KeyAnalyzer
   * @throws Exception in case of any error.
   */
  public KeyAnalyzer() throws Exception
  {
    init();
  }

  /**
   * Perform the calculation
   * @throws Exception in case of any error.
   */
  private void init() throws Exception
  {
    lastSelectedPath = "./";
    try 
    {
      try (BufferedReader in = new BufferedReader(new FileReader(LAST_PATH)))
      {
        lastSelectedPath = in.readLine();
      }
    } 
    catch (Exception e) 
    {
      // Do nothing, if the path does not exist.
    }
    ImageIcon icon = new ImageIcon (loadImage("open.png"));
    btnOpen = new JButton (icon);
    btnOpen.setPreferredSize(BUTTON_SIZE);
    btnOpen.addActionListener(this);
    btnOpen.setBorder(null);
    btnOpen.setRolloverIcon(new ImageIcon (loadImage("openOver.png")));
    btnOpen.setToolTipText("Select a key file");
    
//    source = getSourcePath ();
    display = new Display();
    // Perform the calculation of the equal distribution and the scattering.
    calculate();
    setPreferredSize(SIZE);
    setTitle ("Rel. frequency of byte values");
    img = loadImage("appIcon.png");
    setIconImage(img);
    getContentPane().setLayout(new GridBagLayout());
    // Add the display component to teh content pane.
    getContentPane().add(
            display, 
            new GridBagConstraints( 
                    0, 0, 
                    1, 1, 
                    1.0, 1.0, 
                    GridBagConstraints.CENTER, 
                    GridBagConstraints.BOTH, 
                    new Insets (0,0,0,0), 
                    0, 0));
    //  Let the layout manager organize the view.
    pack ();
    // Center the application on the screen.
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation(screenSize.width / 2 - getWidth () / 2,
            screenSize.height / 2 - getHeight() / 2);
    // Make the window visible.
    setVisible(true);
  }
  
  /**
   * Calculates the relativ occurance of each byte value within the given key 
   * bundle as well as the maximum deviation between these vales.
   * 
   * @throws Exception in case of any issue.
   */
  private void calculate() throws Exception
  {
    if (source != null)
    {
      // Load the key bundle to be examined.
      bundleFile = new File (source);
      key = new byte [(int)bundleFile.length()];
      try (BufferedInputStream bis = 
              new BufferedInputStream(
              new FileInputStream(source))) 
      {
        bis.read(key);
      }
      // Creates an array of 256 integer values to count the occurrence of byte 
      // values within the key bundle.
      int [] counts = new int [256];
      int max = 0;
      byte b;
      // Get the starting index of the pure key bundle values.
      int start = KeyFile.getKeyStartingPoint(key);
      // Count all occurences of the bytes from the specified key bundle.
      for (int j = start; j < key.length; j++)
      {
        // Get the byte at position "j".
        b = key[j];
        // Increment the associated counter.
        counts[HexTool.byteToInt(b)]++;
      }

      // Determine the maximum value of the occurances fornormalizing the values 
      // to the range [0 .. 1].
      for (int c: counts)
      {
        if (c > max)
        {
          max = c;
        }
      }

      // Form relative values with respect to the maximum value.
      for (int jnx = 1; jnx < norm.length; jnx++)
      {
        norm[jnx] = counts[jnx] / (float)max;
      }

      // Use the key quality assessor to calculate the quality values.
      KeyQualityAssessor kqa = new KeyQualityAssessor();
      // Store the results for later use with the display.
      qualityValues = kqa.determineQuality(key);
      display.repaint();
    }
  }
  
  @Override
  protected void processWindowEvent(WindowEvent e)
  {
    if (e.getID() == WindowEvent.WINDOW_CLOSING)
    {
      super.processWindowEvent(e);
      System.exit(0);
    }
    else
    {
      super.processWindowEvent(e);
    }
    
  }

  /**
   * Load the application icon of the Enrypter-X.
   * @return the application icon of the Enrypter-X.
   */
  private Image loadImage(String imgName) 
  {
    BufferedImage img = null;
    // Get the image from the jar resource.
    URL url = KeyAnalyzer.class.getResource("/de/das/encrypter/images/" + imgName);
    try
    {
      InputStream fis = url.openStream();
      img = ImageIO.read(fis);
    }
    catch (Exception e)
    {

    }
    return img;
  }

  /**
   * Gets the source path from the first line of the text file "Probe" in 
   * relativ folder "./Tests".
   * @return the source path.
   */
  private String getSourcePath() 
  {
    String path = null;
    try 
    {
      try (BufferedReader in = new BufferedReader (
              new FileReader (SOURCE_FILE_NAME_CONTAINER)))
      {
        path = in.readLine();
      }
    } 
    catch (Exception e) 
    {
      
    }
    return path;
  }

  @Override
  public void actionPerformed(ActionEvent e) 
  {
    if (e.getSource().equals(btnOpen))
    {
      JFileChooser fc = new JFileChooser (lastSelectedPath);
      fc.setFileFilter(new KeyFileFileFilter("Key file"));
      fc.setAcceptAllFileFilterUsed(false);
      if (fc.showOpenDialog(this) == JFileChooser.APPROVE_OPTION)
      {
        source = fc.getSelectedFile().getAbsolutePath();
        try 
        {
          calculate();
        } 
        catch (Exception x) 
        {
          JOptionPane.showMessageDialog(this, 
                  "Calculation failed", 
                  "Key analyzer", 
                  JOptionPane.ERROR_MESSAGE);
        }
      }
      lastSelectedPath = fc.getCurrentDirectory().getAbsolutePath();
      saveLastPath();
    }
  }

  private void saveLastPath() 
  {
    try 
    {
      try (BufferedWriter out = new BufferedWriter (new FileWriter (LAST_PATH)))
      {
        out.write (lastSelectedPath + "\r\n");
      }
    } 
    catch (Exception e) 
    {
      // Do nothing.
    }
  }
  
  /**
   * A JPanel instance for visualizing the analysis results.
   */
  class Display extends JPanel
  {
    private final int LEFT_BORDER = 30;
    
    private final int RIGHT_BORDER = 30;
    
    private final int TOP_BORDER = 30;
    
    private final int BOTTOM_BORDER = 30;
    // Define the values to be displayed at the horizontal axis.
    private final int [] HORI_TEXTS = {0, 64, 128, 192, 255};

    public Display() 
    {
      init();
    }
    
    private void init()
    {
      setLayout (new GridBagLayout());
      add (btnOpen, new GridBagConstraints(
              0, 0, 1, 1, 1.0, 1.0, 
              GridBagConstraints.NORTHEAST, 
              GridBagConstraints.NONE, 
              new Insets(130,0,0,50), 
              0, 0));
    }
    
    /**
     * Paints the content of the panel.
     * @param g the graphics instance of the JPanel. 
     */
    @Override
    public void paintComponent(Graphics g)
    {
      super.paintComponent(g);
      int hMax = getHeight();
      int y0 = hMax - BOTTOM_BORDER;
      int DY = y0 - TOP_BORDER;
      int y;
      
      // Create white background for the chart area.                     
      g.setColor(Color.WHITE);
      g.fillRect(LEFT_BORDER, TOP_BORDER, 255, y0 - TOP_BORDER);
      // Draw the encrypter logo at the upper right side.
      g.drawImage(img, getWidth() - RIGHT_BORDER - img.getWidth(null), TOP_BORDER, null);
      g.setColor(Color.BLACK);
      
      if (source != null)
      {
        // Underlay the area of the value spread with light blue paint.
        g.setColor(new Color (128, 128, 255, 20));
        g.fillRect(LEFT_BORDER, TOP_BORDER, 255, (int) (DY * (1 - qualityValues[1])));
        // Draw a blue line at the average frequency.
        y = (int)(y0 - DY * qualityValues[0]);
        g.setColor(Color.BLUE);
        g.drawLine(LEFT_BORDER, y, LEFT_BORDER + 255, y);
        // Select the color for painting the chart and information.
        g.setColor(Color.BLACK);
        // Paint the relative frequency as a curve.
        for (int i = 1; i < norm.length - 1; i++)
        {
          g.drawLine(
                  LEFT_BORDER + i, (int)(y0 - DY * norm[i]), 
                  LEFT_BORDER + i + 1, (int) (y0 - DY * norm[i + 1]));
        }
      }
      
      // Draw the axis.
      g.drawLine(LEFT_BORDER, y0, LEFT_BORDER + 255, y0);
      g.drawLine(LEFT_BORDER, y0, LEFT_BORDER, TOP_BORDER);
      
      FontMetrics fm = g.getFontMetrics();
      String str;
      int x;
      int x0;
      y = y0 + fm.getHeight();
      // Write the texts on the horizontal axis.
      for (int j = 0; j < HORI_TEXTS.length; j++)
      {
        str = Integer.toString(HORI_TEXTS[j]);
        x0 = LEFT_BORDER + HORI_TEXTS[j];
        x = x0 - fm.stringWidth(str) / 2;
        g.drawLine(x0, y0 - 2, x0, y0 + 2);
        g.drawString(str, x, y);
      }
      
      // Write the texts on the vertical axis.
      x = LEFT_BORDER - 2;
      for (int j = 1; j <= 10; j++)
      {
        str = Tools.getFormatted("%1$3.1f", j / 10.0);
        y = y0 - (int)(DY * j / 10.0);
        g.drawLine(x, y, x + 4, y);
        g.drawString(str, x - 4 - fm.stringWidth(str), y + fm.getHeight() / 2);
      }
      
      // Specify left side and top of the area for the text output.
      int leftTextBorder = getWidth() - 120;
      int topTextBorder = getHeight() - 100;
      
      if (source != null)
      {
        g.drawString ("Analyzed key bundle:", leftTextBorder, topTextBorder - 6 * fm.getHeight());
        g.drawString(bundleFile.getName(), 
                leftTextBorder, topTextBorder - 5 * fm.getHeight());

        g.drawString ("Key bundle size:", leftTextBorder, topTextBorder - 3 * fm.getHeight());
        g.drawString(Integer.toString (Math.round(KeyFile.getPureKeyData(key).length / 1024.0f)) + "kB", 
                leftTextBorder, topTextBorder - 2 * fm.getHeight());

        g.drawString ("Equal distribution", leftTextBorder, topTextBorder);
        g.drawString(Tools.getFormatted("%1$6.4f",qualityValues[0]), 
                leftTextBorder, topTextBorder + fm.getHeight());

        g.drawString ("Scattering", leftTextBorder, topTextBorder + fm.getHeight() * 3);
        g.drawString(Tools.getFormatted("%1$6.4f",1 - qualityValues[1]), 
                leftTextBorder, topTextBorder + fm.getHeight() * 4);
      }
    }
  }
}
