package de.das.encrypter.examples;

import de.das.encrypter.model.KeyFileLocation;
import de.das.encrypter.model.KeyFiles;
import de.das.encrypter.model.KeyFilesSearchReceiver;

import de.das.encrypter.processors.KeyFilesSearcher;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;

import java.awt.event.WindowEvent;

import java.io.File;

import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

/**
 * This example illustrates the use of the key bundle file searcher. It uses
 * a small GUI for the demonstration. The background of the GUI turns green when 
 * the searcher performs its search. The result is then output in the text 
 * window. 
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class KeySearchingExample extends JFrame implements KeyFilesSearchReceiver
{
  // The dimension of the application window.
  private final Dimension SIZE = new Dimension (480, 480);
  
  // An instance of the inner class used for the illustration.
  private final Display display = new Display();
  
  // The hash map that will hold and manage the key bundle files. 
  private final KeyFiles keyFiles = new KeyFiles();
  
  public static void main (String [] args)
  {
    try 
    {
      new KeySearchingExample().execute();
    } 
    catch (Exception e) 
    {
      e.printStackTrace();
    }
  }
  
  // Flag to show the searcher that he can now perform his task.
  private final boolean searchingAllowed = true;
  
  // Flag set by the searcher while performing his search.
  private boolean searching;
  
  // A list with "bundle" folder found by the searcher.
  private ArrayList < File > bundleFolders;
  
  // A list with information about all found key bundles.
  private ArrayList < KeyFileLocation > keyFileLocations;

  private void execute() throws Exception
  {
    // The size of the GUI.
    setPreferredSize(SIZE);
    // The application title.
    setTitle("Keys searching example");
    
    getContentPane().setLayout(new GridBagLayout());
    getContentPane().add(display, new GridBagConstraints(
            0, 0, 1, 1, 1.0, 1.0,
            GridBagConstraints.CENTER,
            GridBagConstraints.BOTH,
            new Insets(0,0,0,0),
            0, 0));
    pack ();
    
    // Centers the graphical user interface on the screen.
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation(screenSize.width / 2 - getWidth () / 2,
            screenSize.height / 2 - getHeight() / 2);
    
    // This creates the key files searcher, tells it that we want the results 
    // and starts it and that's all. From now on it will provide every 10 
    // seconds the actual search result.
    KeyFilesSearcher.createInstance(this);
    
    // Makes the GUI visible.
    setVisible(true);
  }

  /**
   * Method of the "KeyFilesSearchReceiver" interface outputs a list of all key 
   * bundles found during the last search, with which the application can 
   * synchronize its current list.
   * 
   * @param hashMap the list with all key bundle files found by the last search.
   */
  @Override
  public void synchronizeKeyFilesList(KeyFiles hashMap) 
  {
    Iterator it = hashMap.keySet().iterator();
    Long id;
    while (it.hasNext())
    {
      id = (Long) it.next();
      getKeyFiles().put(id, hashMap.get(id));
    }
    display.update();
  }

  /**
   * Method of the "KeyFilesSearchReceiver" interface through which the searcher 
   * can access the application's list.
   * 
   * @return the list of key bundle files.
   */
  @Override
  public KeyFiles getKeyFiles() 
  {
    return keyFiles;
  }

  /**
   * Method of the "KeyFilesSearchReceiver" interface through which the searcher 
   * can pass a list with information about all found key bundle files.
   * 
   * @param keyFileLocations 
   */
  @Override
  public void setKeyFileLocations(ArrayList < KeyFileLocation > keyFileLocations) 
  {
    this.keyFileLocations = keyFileLocations;
  }

  /**
   * Method of the "KeyFilesSearchReceiver" interface through which the searcher 
   * can tell the application that it is now performing a search.
   * 
   * @param b <b>true</b>, if the searcher is performing a search.
   */
  @Override
  public void setKeyFileSearching(boolean b) 
  {
    searching = b;
    display.update();
  }

  /**
   * Method of the interface "KeyFilesSearchReceiver" via which the searcher 
   * passes the list of all found "bundle" folders.
   * 
   * @param bundleFolders a list with the found "bundle" folders.
   */
  @Override
  public void setBundleFolders(ArrayList < File > bundleFolders) 
  {
    this.bundleFolders = bundleFolders;
    display.update();
  }

  /**
   * Method of the interface "KeyFilesSearchReceiver" via which the searcher can 
   * request whether it is now allowed to perform a search.
   * 
   * @return <b>true</b>, if the searcher is allowed to search now.
   */
  @Override
  public boolean isSearchingAllowed() 
  {
    return searchingAllowed;
  }
  
  /**
   * Overwritten method to terminate the application if the window is closed.
   * 
   * @param e a window event.
   */
  @Override
  protected void processWindowEvent(WindowEvent e)
  {
    if (e.getID() == WindowEvent.WINDOW_CLOSING)
    {
      super.processWindowEvent(e);
      System.exit(0);
    }
    else
    {
      super.processWindowEvent(e);
    }
  }
  
  /**
   * An inner class to visualize the searcher's activity and results.
   */
  class Display extends JPanel
  {
    // A text area to show the dearching result.
    private final JTextArea taResults = new JTextArea();
    
    public Display()
    {
      init();
    }
    
    private void init()
    {
      taResults.setEditable(false);
      
      JScrollPane scroller = new JScrollPane(
              taResults, 
              JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, 
              JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
      setLayout(new GridBagLayout());
      add (scroller,new GridBagConstraints(
            0, 0, 1, 1, 1.0, 1.0,
            GridBagConstraints.CENTER,
            GridBagConstraints.BOTH,
            new Insets(20,20,20,20),
            0, 0));

    }
    
    @Override
    public void paintComponent(Graphics g)
    {
      super.paintComponent(g);
      paintBackground(g);
    }
    
    void update()
    {
      String results = "Bundle folders found:\n" + 
          (bundleFolders == null ? 
              "\n" : 
              (bundleFolders.toString() + "\n\nKey file locations:\n"));
      
      if (keyFileLocations != null)
      {
        ArrayList < KeyFileLocation > list = keyFileLocations;
        for (KeyFileLocation kfl: list)
        {
          results = results + 
                  expandTo(kfl.getKeyId(), 16) + 
                  " " + 
                  kfl.getKeyName() + 
                  "  --->  " + 
                  kfl.getKeyLocation().replace("\\\\", "\\") + "\n\r";
        }
       }
      taResults.setText(results);
      repaint();
    }

  private String expandTo(long id, int n)
  {
    String s = Long.toString(id);
    while (s.length() < n)
    {
      s = "0" + s;
    }
    return s;
  }

    private void paintBackground(Graphics g) 
    {
      if (searching)
      {
        g.setColor(new Color (120, 255, 120));
        g.fillRect(0, 0, getWidth(), getHeight());
      }
    }
  }
}
