package de.das.encrypter.examples;

import de.das.encrypter.model.KeyFile;

import de.das.encrypter.tools.HexTool;

import java.io.BufferedInputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;

import java.util.ArrayList;
import java.util.Arrays;

import java.util.HashMap;

/**
 * This example shows how uniformly distributed the random values of a key 
 * bundle are. A search is made in the key ring for the occurrence of identical 
 * patterns. First, patterns consisting of the three consecutive bytes are 
 * formed and compared with all three-byte patterns of the key ring. Then the 
 * patterns with four consecutive bytes and so on. The number of bytes in the 
 * pattern is increased until no more identical patterns are found.
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class PatternChecker 
{
  // Start value of the pattern sizes that will be incremented in a loop until 
  // no more multiple occurences are found. For each pattern size, the number of 
  // multiple occurrences is searched.  
  private int patternSize = 3;
 
  public static void main (String [] args)
  {
    try
    {
      new PatternChecker().execute();
    } 
    catch (Exception e)
    {
      e.printStackTrace();
    }
    System.exit(0);
  }

  private void execute() throws Exception
  {
    // Specify a file for the vcreation of a key bundle.
    File f = new File ("./16.000-1");
    // Create it.
    KeyFile.create(f, 16000);
    // Create a hash map to collect the results of the pattern searches.
    // The keys of this hash map are the pattern and the values array lists of 
    // string with the references "index of pattern"-"index of further 
    // occurance".
    HashMap < String, ArrayList < String > > results = new HashMap <> ();
    
    // Now load the key bundle into a byte array.
    byte [] key = new byte [(int)f.length()];
    try (BufferedInputStream bis = 
            new BufferedInputStream(
            new FileInputStream(f))) 
    {
      bis.read(key);
    }
    
    // Get the starting index of the pure key bundle data behinde the header.
    int start = KeyFile.getKeyStartingPoint(key);
    
    // Define a string to contain the pattern as a hex string to be used as the
    // key in the results map.
    String patternString;
    
    // Create a loop in which the pattern size is incremented until no more 
    // duplicate patterns can be found.
    do
    {
      // Clear the results map.
      results.clear();
      // Create a byte array for the new pattern size.
      byte [] pattern = new byte [patternSize];
      // Create a byte array for the patterns to compare to each pattern.
      byte [] check = new byte [patternSize];
      // Now start a loop and get each pattern from each index.
      for (int i = start; i < key.length - patternSize; i++)
      {
        // Copy the pattern at index "i" to the pattern array.
        System.arraycopy(key, i, pattern, 0, patternSize);
        // Build a hex string from the pattern to use it as the key of the 
        // result list.
        patternString = HexTool.byteArrayToString(pattern);
        // If this pattern not yet is contained in the result list start 
        // comparison with the rest of the key bundle.
        if (!results.containsKey(patternString))
        {
          // Create a new array list to collect the detected occurances.
          ArrayList < String > references = new ArrayList <> ();
          // Now compare the pattern with all pattern behinde since matches with 
          // pattern before have already been found if they exist. 
          for (int j = i + 1; j < key.length - patternSize; j++)
          {
            // Take the pattern to compare.
            System.arraycopy(key, j, check, 0, patternSize);
            // Check if the pattern match.
            if (Arrays.equals(pattern, check))
            {
              // Add the reference between the two pattern to the results.
              references.add (Integer.toString(i) + "-" + Integer.toString(j));
            }
          }
          // Check, if the references is not empty and the results map does not 
          // yet contains the key "patternString".
          if (!references.isEmpty() && !results.containsKey(patternString))
          {
            // Add the results for key "patternString".
            results.put(patternString, references);
          }
        }
      }
      // All comparisons have been performed.
      if (!results.isEmpty())
      {
        // Any results are available. So create a report file for the results
        // with the current pattern size.
        try(
          BufferedWriter out = 
              new BufferedWriter (
              new FileWriter (
                      f.getAbsolutePath() + 
                      "-PatternCheck-" + 
                      Integer.toString(patternSize) + 
                      ".txt")))
        {
          // Write the results into a text file.
          for (String p: results.keySet())
          {
            ArrayList < String > refs = results.get(p);
            out.write (
                    p + 
                    ": " + 
                    HexTool.fillLeadingPositions(
                            Integer.toString(refs.size()), 
                            3, 
                            ' '));
            out.write ("  ");
            for (String s : refs)
            {
              out.write (s + "; ");
            }
            out.write("\r\n");
          }
        }
      }
      // Increment the pattern size and start the next cycle
      patternSize++;
    }
    // Start the next cycle only if the previous run has results.
    // If no results were obtained with the previous pattern size, there will be 
    // no results for larger pattern sizes.
    while (!results.isEmpty());
  }

}
