package de.das.encrypter.examples;

import de.das.encrypter.model.KeyFile;

import de.das.encrypter.tools.HexTool;

import java.io.BufferedInputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;

/**
 * This analysis looks for the occurrence of the same patterns between every two 
 * key bundles of the same length. The pattern sizes are incremented starting 
 * with three until no more equal patterns are found between the two key 
 * bundles. The index references are stored together with the respective pattern 
 * as a result and written to a text file at the end of the search. Twenty key 
 * bundles located in the relative folder "./Tests/PatternCompare" with the 
 * names "TestKey_n" are examined, where n are the numbers 0 to 19.
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class PatternCompare 
{
  private final String SOURCE = ".\\Tests\\PatternCompare\\TestKey_";
  
  private final int START_PATTERN_SIZE = 3;
  
  private int patternSize;
 
  public static void main (String [] args)
  {
    try
    {
      PatternCompare pc = new PatternCompare();
      for (int i0 = 0; i0 < 20; i0++)
      {
        for (int i1 = i0 + 1; i1 < 20; i1++)
        {
          pc.execute(i0, i1);
        }
      }
    } 
    catch (Exception e)
    {
      e.printStackTrace();
    }
    System.exit(0);
  }

  /**
   * Performs an examination between the two key bundles with the endings "i0" 
   * and "i1".
   * 
   * @param i0 extension of one file.
   * @param i1 extension of the other file.
   * 
   * @throws Exception in case of an error.
   */
  private void execute(int i0, int i1) throws Exception
  {
    // Preset the pattern size with the starting value.
    patternSize = START_PATTERN_SIZE;
    // Create an instance of a hash map to collect the found matches.
    HashMap < String, ArrayList < String > > results = new HashMap <> ();
    
    // Load the values of the one key bundle.
    File fSource = new File (SOURCE + Integer.toString(i0));
    byte [] key = new byte [(int)fSource.length()];
    try (BufferedInputStream bis = 
            new BufferedInputStream(
            new FileInputStream(fSource))) 
    {
      bis.read(key);
    }

    // Load the values of the other key bundle.
    File f = new File (SOURCE + Integer.toString(i1));
    byte [] keyComp = new byte [(int)f.length()];
    try (BufferedInputStream bis = 
            new BufferedInputStream(
            new FileInputStream(f))) 
    {
      bis.read(keyComp);
    }
    
    // Get the start index of the pure key bundle values.
    int startKey = KeyFile.getKeyStartingPoint(key);
    String patternString;
    
    // Do the comparison with inreasing pattern size until no more matches 
    // occur.
    do
    {
      // Clear the results of the previous search.
      results.clear();
      // Create byte arrays as buffers for the pattern to compare.
      byte [] pattern = new byte [patternSize];
      byte [] check = new byte [patternSize];
      
      // Use the pattern at each index of the key bundle as a test pattern.
      for (int i = startKey; i < key.length - patternSize; i++)
      {
        // Copy the pattern bytes to the pattern buffer.
        System.arraycopy(key, i, pattern, 0, patternSize);
        // Create a hex string from the pattern to be used as the key of the 
        // result map.
        patternString = HexTool.byteArrayToString(pattern);
        
        // Starts the comparison only if there are no results in the result map 
        // for the pattern yet.
        if (!results.containsKey(patternString))
        {
          // Create an array list to collect the found references.
          ArrayList < String > references = new ArrayList <> ();
          // Fetches the pattern at all indices larger than the current position 
          // of the test pattern.
          for (int j = i + 1; j < key.length - patternSize; j++)
          {
            // Copy the pattern to the check buffer.
            System.arraycopy(keyComp, j, check, 0, patternSize);
            // Compare the pattern.
            if (Arrays.equals(pattern, check))
            {
              // If they match build a reference string and add it to the 
              // references list.
              references.add (Integer.toString(i) + "-" + Integer.toString(j));
            }
          }
          // If any matches are found put the references list to the result map.
          if (!references.isEmpty())
          {
            results.put(patternString, references);
          }
        }
      }
      // When matches are found, the results are written to a text file.
      if (!results.isEmpty())
      {
        try(BufferedWriter out = new BufferedWriter (
                new FileWriter (
                  f.getAbsolutePath() + 
                  "-" + 
                  fSource.getName() + 
                  "-PatternCompare-" + 
                  Integer.toString(patternSize) + 
                  ".txt")))
        {
          for (String p: results.keySet())
          {
            ArrayList < String > refs = results.get(p);
            out.write (
                p + 
                ": " + 
                HexTool.fillLeadingPositions(
                  Integer.toString(refs.size()), 
                  3, 
                  ' '));
            out.write ("  ");
            for (String s : refs)
            {
              out.write (s + "; ");
            }
            out.write("\r\n");
          }
        }
      }
      // Increment the pattern size for the next run.
      patternSize++;
    }
    // Perform the next run if the previous run gave results.
    while (!results.isEmpty());
  }

}
