package de.das.encrypter.model;

import de.das.encrypter.tools.CRC;
import de.das.encrypter.tools.HexTool;
import de.das.encrypter.tools.KeyQualityAssessor;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.RandomAccessFile;

import java.util.ArrayList;
import java.util.Objects;
import java.util.Random;

/**
 * Class, which represents a key with its identifier and current path to
 * its location.
 * Instances of this class are created when keys are searched for and found in 
 * the system. However, the class also contains many static methods for 
 * generating new keys and checking the validity of given keys. 
 * <br><br>
 * The key itself is represented by a byte array headed with a header and filled 
 * with random bytes of data. And it is important to note that these are true 
 * random numbers and not pseudo-random numbers.<br><br>
 * The header structure is the key signature "KEYFILE_XXXXXXXXXXXXXXXXNNnnnnnnnn" 
 * where "XXXXXXXXXXXXXXX" is a unique key file identifier represented as an 
 * hexadecimal string, "NN" is the number of following bytes as a 16 Bit integer 
 * (little endian) representing the name of the key "nnnnnnn".
 * 
 * @author Frank Keldenich
 */
public class KeyFile implements Key
{
  /**
   * The length of the key header for key signature and key ID.
   */
  public static final int HEADER_LENGTH = 24;
  
  /**
   * The length of the CRC value appended to the keys.
   */
  public static final int CRC_LENGTH = 4;
  
  /**
   * The length of the key header plus two for the bytes for the length of the 
   * following key name.
   */
  public static final int HEADER_LENGTH_PLUS_NAME_AMOUNT = 26;
  
  private static int lastEntryPointAdd;
  
  private final long id;
  
  private final String path;
  
  private final int length;
  
  private Integer pureKeyDataLength = null;
  
  private final String driveLetter;

  private final String keyName;
  
  private static Random random;
  
  private byte [] key;
  
  private Integer used = null;


  /**
   * Creates a new instance of the class KeyFile.
   * 
   * @param id the ID of the associated key.
   * @param key the associated key itself.
   * @param driveLetter the letter of the drive on which it is located.
   */
  public KeyFile(long id, byte [] key, String driveLetter)
  {
    this.id = id;
    this.key = key;
    this.driveLetter = driveLetter;
    this.length = key.length;
    this.path = null;
    this.keyName = getKeyName(key);
  }

  /**
   * Creates a new instance of the class KeyFile.
   * 
   * @param id the ID of the associated key.
   * @param path the path where the key is located.
   * @param length the length of the key.
   */
  public KeyFile(long id, String path, int length)
  {
    this (id, path, length, null);
  }

  /**
   * Creates a new instance of the class KeyFile.
   * 
   * @param id the ID of the associated key.
   * @param path the path where the key is located.
   * @param length the length of the key.
   * @param key the byte array containing the key with header.
   */
  public KeyFile(long id, String path, int length, byte [] key)
  {
    this.id = id;
    this.path = path;
    this.length = length;
    driveLetter = path.substring(0, 1);
    keyName = new File(path).getName();
    this.key = key;
  }

  /**
   * Returns the ID of the associated key.
   * @return the ID of the associated key.
   */
  public long getId()
  {
    return id;
  }

  /**
   * Returns the path of the location of the associated key.
   * @return the path of the location of the associated key.
   */
  public String getPath()
  {
    return path;
  }

  /**
   * Returns the length of the associated key. It is the length of the key 
   * including the header.
   * 
   * @return the length of the associated key.
   */
  public int getLength()
  {
    return length;
  }
  
  /**
   * Checks, whether the given data starting part starts with the keyword 
   * "KEYFILE_" and decodes the following "XXXXXXXXXXXXXXXX" bytes as the key 
   * identifier.<br><br>
   * "XXXXXXXXXXXXXXXX" is the string representation of the hex identifier.
   * 
   * @param dataStart a byte array with the starting part of encrypted data.
   * @return the key file identifier or <b>null</b>, if the data start does 
   * not contain valid information.
   */
  public static Long getKeyFileIdentifier (byte [] dataStart)
  {
    Long id = null;
    if (dataStart.length >= Key.KEY_HEADER_LENGTH)
    {
      byte [] keyword = new byte [Key.KEY_SIGNATURE_PREFIX.length()];
      System.arraycopy(dataStart, 0, keyword, 0, keyword.length);
      if (new String (keyword).equals(Key.KEY_SIGNATURE_PREFIX))
      {
        byte [] idBytes = new byte [16];
        System.arraycopy(dataStart, Key.KEY_SIGNATURE_PREFIX.length(), 
                idBytes, 0, 16);
        try
        {
          id = Long.parseLong(new String(idBytes), 16);
        } 
        catch (NumberFormatException e)
        {
          // Nothing. If parsing fails the resulting "id" will be <b>null</b>.
        }
      }
    }
    return id;
  }
  
  /**
   * Checks, whether the content of the given file starts with the keyword 
   * "KEYFILE_" and decodes the following "XXXXXXXXXXXXXXXX" bytes as the key 
   * identifier.<br><br>
   * "XXXXXXXXXXXXXXXX" is the string representation of the hex identifier.
   * 
   * @param f a containing encrypted data.
   * @return the key identifier or <b>null</b>, if the file does not contain 
   * valid information.
   */
  public static Long getKeyFileIdentifier (File f)
  {
    Long id;
    try 
    {
      byte [] header = new byte [Key.KEY_HEADER_LENGTH];
      try (RandomAccessFile raf = new RandomAccessFile(f, "r")) 
      {
        raf.read(header);
      }
      id = getKeyFileIdentifier(header);
    } 
    catch (Exception e) 
    {
      id = null;
    }
    return id;
  }
  
  /**
   * Generates and returns a string consisting of the key name followed by 
   * information about the drive it is on and its length and how many virtual 
   * keys are still available.
   * 
   * @return a string consisting of the key name followed by information about 
   * the drive it is on and its length.
   */
  @Override
  public String toString ()
  {
    int pureLength = 0;
    try 
    {
      pureLength = getPureKeyDataLength();
    } 
    catch (Exception e) 
    {
      
    }
    return keyName + " (" + driveLetter + " - " + 
            (length <= KB ? Integer.toString(length) + "B)" : (
            (length <= MB ? 
            Integer.toString(length / 1024) + "kB)" : 
            Integer.toString(length / 1024 / 1024) + "MB)"))) +
            (used != null ? 
            " [" + insertThousandsSeparators(Integer.toString(pureLength - used)) + "]" : 
            "");
  }
  
  /**
   * Creates a new key using the bytes of a given file.<br><br>
   * It is recommended not to use such keys if the encryption is to be 100%
   * secure.
   * 
   * @param target the file that should contain the key.
   * @param contentFile the path to a file to be used as a key.
   * @throws Exception in case of any error.
   */
  public static void create(File target, String contentFile) throws Exception
  {
    File cf = new File (contentFile);
    byte [] content = new byte [(int)cf.length()];
    try (BufferedInputStream in = new BufferedInputStream(
            new FileInputStream(cf))) 
    {
      in.read(content);
    }
    for (int i = 0; i < content.length; i++)
    {
      // Make sure no zero value is contained in the key bundle.
      if (content[i] == 0)
      {
        content[i] = (byte) random.nextInt(1, 256);
      }
    }
    byte [] key = create (content, target.getName());
    try (BufferedOutputStream bos = new BufferedOutputStream(
            new FileOutputStream(target))) 
    {
      bos.write(key);
    }
  }
  
  /**
   * Creates a new key using the given byte array for the key data and the given 
   * name for the key name. It is recommended not to use such keys if the 
   * encryption shall be 100% secure and the given data are not true randomly.
   * 
   * @param data the given byte array.
   * @param name the name of the key.
   * @return a key to be used for encryption.
   */
  public static byte [] create(byte [] data, String name)
  {
    long crc = CRC.getCRC_32(data);

    String header = Key.KEY_SIGNATURE_PREFIX + 
            HexTool.toTwoHexQuad((int)((crc >> 32) & 0xFFFFFFFF)) +
            HexTool.toTwoHexQuad((int)(crc & 0xFFFFFFFF));
    header = header + HexTool.toHex((byte) name.getBytes().length) + name;
    byte [] headerBytes = header.getBytes();
    
    byte [] key = new byte [data.length + headerBytes.length + CRC_LENGTH];
    System.arraycopy(headerBytes, 0, key, 0, headerBytes.length);
    System.arraycopy(data, 0, key, headerBytes.length, data.length);

    long crcValue = CRC.getCRC_32(key, 4);
    HexTool.insertIntBigEndianAt(crcValue, key.length - 4, key);
    return key;
  }
  
  /**
   * Creates a key file with leading key signature "KEYFILE_XXXXXXXXXXXXXXXXNNnnnnnnnn" 
   * where "XXXXXXXXXXXXXXX" is a unique key file identifier. 
   * "XXXXXXXXXXXXXXXX" is the string representation of the hex identifier,
   * "NN" is the number of following bytes representing the name of the key 
   * "nnnnnnn".
   * 
   * @param target the file that should contain the key.
   * @param length the request length of the key.
   * @throws Exception in case of any error.
   */
  public static void create(File target, int length) throws Exception
  {
    create (target, length, System.currentTimeMillis(), false);
  }
  
  /**
   * Creates a key file with leading key signature "KEYFILE_XXXXXXXXXXXXXXXXNNnnnnnnnn" 
   * where "XXXXXXXXXXXXXXX" is a unique key file identifier. 
   * "XXXXXXXXXXXXXXXX" is the string representation of the hex identifier,
   * "NN" is the number of following bytes representing the name of the key 
   * "nnnnnnn".
   * 
   * @param target the file that should contain the key.
   * @param length the request length of the key.
   * @throws Exception in case of any error.
   */
  public static void create(File target, int length, boolean balanced) throws Exception
  {
    create (target, length, System.currentTimeMillis(), balanced);
  }

  /**
   * Creates a file with a key with leading key signature "KEYFILE_XXXXXXXXXXXXXXXXNNnnnnnnnn" 
   * where "XXXXXXXXXXXXXXX" is a unique key file identifier. 
   * "XXXXXXXXXXXXXXXX" is the string representation of the hex identifier,
   * "NN" is the number of following bytes representing the name of the key 
   * "nnnnnnn".
   * 
   * @param target the file that will contain the key.
   * @param length the request length of the key.
   * @param seed a start value (type long) for the key development.
   * @param balanced a flag that instructs to use the balancer.
   * @throws Exception in case of any error.
   */
  public static void create(File target, int length, long seed, boolean balanced) throws Exception
  {
    byte [] key = create (target.getName(), length, seed, balanced);
    try (BufferedOutputStream bos = new BufferedOutputStream(
            new FileOutputStream(target))) 
    {
      bos.write(key);
    }
  }
  
  /**
   * Creates a key with leading key signature "KEYFILE_XXXXXXXXXXXXXXXXNNnnnnnnnn" 
   * where "XXXXXXXXXXXXXXX" is a unique key file identifier. 
   * "XXXXXXXXXXXXXXXX" is the string representation of the hex identifier,
   * "NN" is the number of following bytes representing the name of the key 
   * "nnnnnnn".
   * 
   * @param name the name of the key (must be a valid file name.
   * @param length the requested key length not smaller than 128.
   * @param seed a random start value for the creation algorithm.
   * @param balanced a flag that instructs to use the balancer.
   * @return a byte array containing the new key.
   * @throws Exception in case of any error.
   */
  public static byte [] create (String name, int length, long seed, boolean balanced) throws Exception
  {
    if (length < 128)
    {
      throw new Exception ("Too small");
    }
    random = new Random(System.currentTimeMillis());
    String header = Key.KEY_SIGNATURE_PREFIX + 
            HexTool.toTwoHexQuad((int)((seed >> 32) & 0xFFFFFFFF)) +
            HexTool.toTwoHexQuad((int)(seed & 0xFFFFFFFF));
    header = header + HexTool.toHex((byte) name.getBytes().length) + name;
    byte [] headerBytes = header.getBytes();
    byte [] key = new byte [length + headerBytes.length];
    System.arraycopy(headerBytes, 0, key, 0, headerBytes.length);
    for (int inx = header.length(); inx < key.length; inx++)
    {
      seed = develope(inx, seed, key);
    }
    long crcValue = CRC.getCRC_32(key, 4);
    HexTool.insertIntBigEndianAt(crcValue, key.length - 4, key);
    if (balanced)
    {
      key = KeyFile.harmonicallyBalance(key, 1);
    }
    
    return key;
  }
 
  /**
   * Creates a new random byte for a key at a given position using a given 
   * random seed. It creates a new seed for the generation of the next byte and 
   * although an instance of the "Random" class is initially used for the 
   * generation, which only generates pseudo-random numbers, an exclude-or 
   * function with a value supplied by the System.nanoTime() method turns it 
   * into a true random number, since the values supplied depend on different 
   * random factors (times of the call, processor speed, current utilization).
   * 
   * @param inx the position for the next created byte in the key.
   * @param seed the seed to generate a random byte.
   * @param key the byte array that should contain the key.
   * @return a new seed for the next step. 
   */
  private static long develope (int inx, long seed, byte[] key)
  {
    key[inx] = (byte) ((seed >> 16) & 0x00FF);
    key[inx] = key[inx] == 0 ? (byte) random.nextInt(1, 256) : key[inx];
    seed = seed ^ ((seed >> 1) + 1);
    seed = seed + random.nextLong();
    seed = seed ^ System.nanoTime();
    return seed;
  }
  
  /**
   * Creates a new key bundle by using pseudo random numbers. Their advantage 
   * as well as their disadvantage is their reproducibility.
   * 
   * @param f the file that shall contain the key bundle.
   * @param length the required key bundle length.
   * @param seed a seed for the developing of different values.
   * @throws Exception in case of any error.
   */
  public static void createPseudoRandomKey (File f, int length, long seed) throws Exception
  {
    createPseudoRandomKey(f, length, seed, false);
  }
  
  /**
   * Creates a new key bundle by using pseudo random numbers. Their advantage 
   * as well as their disadvantage is their reproducibility.
   * 
   * @param f the file that shall contain the key bundle.
   * @param length the required key bundle length.
   * @param seed a seed for the developing of different values.
   * @param balanced a flag that instructs to use the balancer.
   * @throws Exception in case of any error.
   */
  public static void createPseudoRandomKey (File f, int length, long seed, boolean balanced) throws Exception
  {
    byte [] pureKeyData = new byte [length];
    Random random = new Random(seed);
    random.nextBytes (pureKeyData);
    for (int i = 0; i < pureKeyData.length; i++)
    {
      // Make sure no zero value is contained in the key bundle.
      if (pureKeyData[i] == 0)
      {
        pureKeyData[i] = (byte) random.nextInt(1, 256);
      }
    }
    byte [] keyBundle =  create (pureKeyData, f.getName());
    if (balanced)
    {
      keyBundle = KeyFile.harmonicallyBalance(keyBundle, 1);
    }
    try (BufferedOutputStream out = 
            new BufferedOutputStream(
                    new FileOutputStream(f)))
    {
      out.write(keyBundle);
    }
  }
  
  /**
   * Compares a given object with this key file object and returns <b>true</b>,
   * if it is.
   * 
   * @param obj a given object.
   * @return <b>true</b>, if it is.
   */
  @Override
  public boolean equals (Object obj)
  {
    boolean equ = obj != null && obj instanceof KeyFile;
    if (equ)
    {
      KeyFile kf = (KeyFile)obj;
      equ = kf.id == id;
      equ = equ && ((kf.getPath() == null && getPath() == null) || 
              (kf.getPath() != null && kf.getPath().equals(getPath())));
      equ = equ && kf.driveLetter.equals(driveLetter);
    }
    return equ;
  }

  /**
   * Returns a hash code value for the object.
   * @return a hash code value for the object.
   */
  @Override
  public int hashCode()
  {
    int hash = 7;
    hash = 83 * hash + (int) (this.id ^ (this.id >>> 32));
    hash = 83 * hash + Objects.hashCode(this.path);
    return hash;
  }
  
  /**
   * Checks whether the specified byte array contains a valid key.
   * 
   * @param key a given byte array. It is necessary that the byte array contains 
   * not only the beginning of the key (header) but the whole key, because a CRC 
   * check is also performed and the CRC value is located at the end of the key.
   * 
   * @return <b>true</b> if the specified byte array contains a valid key.
   */
  public static boolean isValidKey (byte [] key)
  {
    return isValidKey(key, false);
  }
  
  /**
   * Checks whether the specified byte array contains a valid key.
   * 
   * @param key a given byte array assumed to be a key. Only the start of the 
   * key (header) can be used for checking if the CRC check is omitted. 
   * @param withoutCrcCheck a flag which determines whether a CRC check should 
   * be performed or not.
   * 
   * @return <b>true</b> if the specified byte array contains a valid key.
   */
  public static boolean isValidKey (byte [] key, boolean withoutCrcCheck)
  {
    byte [] keyStart = new byte [Key.KEY_HEADER_LENGTH];
    boolean crcOk = true;
    System.arraycopy(key, 0, keyStart, 0, keyStart.length);
    boolean valid = getKeyFileIdentifier(keyStart) != null;
    if (!withoutCrcCheck)
    {
      long crc = CRC.getCRC_32(key, 4);
      long fileCrc = HexTool.byteToUintBigEndian(key.length - 4, key);
      crcOk = crc == fileCrc;
    }
    return valid && (crcOk || withoutCrcCheck);
  }
  
  /**
   * Checks whether the specified file contains a valid key. The check includes
   * the CRC32 check.
   * 
   * @param f a given file that is assumed to contain a key.
   * @return <b>true</b> if the specified file contains a valid key.
   */
  public static boolean isValidKey (File f)
  {
    return isValidKey(f, false);
  }
  
  /**
   * Checks whether the specified file contains a valid key. If the key itself 
   * is not needed, but only to check whether the file contains a valid key, 
   * only the beginning of the file can be used for checking if the CRC check 
   * is omitted, so that only the key headers are used for performance reasons.
   * 
   * @param f a given file that is assumed to contain a key.
   * @param withoutCheck a flag which determines whether a CRC check should 
   * be performed or not.
   * @return <b>true</b> if the specified file contains a valid key.
   */
  public static boolean isValidKey (File f, boolean withoutCheck)
  {
    boolean valid;
    try 
    {
      byte [] data = new byte [withoutCheck ? Key.KEY_HEADER_LENGTH : (int)f.length()];
      try (RandomAccessFile raf = new RandomAccessFile(f, "r")) 
      {
        raf.read(data);
      }
      valid = isValidKey(data, withoutCheck);
    } 
    catch (Exception e) 
    {
      valid = false;
    }
    return valid;
  }

  /**
   * Returns the CRC value of the key by forming it from its bytes except the 
   * last four.
   * 
   * @return the CRC value of the key.
   */
  public int getCrcValue()
  {
    int crc = -1;
    try
    {
      crc = (int) CRC.getCRC_32(getKey(), 4);
    } 
    catch (Exception e)
    {
    }
    return crc;
  }

  /**
   * Extracts and returns the key name from the header of the given key.
   * @param key a given key or key header.
   * 
   * @return returns the key name from the header.
   */
  public static String getKeyName(byte[] key)
  {
    byte [] amountBytes = new byte [2]; 
    System.arraycopy(key, HEADER_LENGTH, amountBytes, 0, amountBytes.length);
    byte nameLength = HexTool.parseHex(new String(amountBytes));
    byte [] nameBytes = new byte [nameLength];
    System.arraycopy(key, HEADER_LENGTH_PLUS_NAME_AMOUNT, nameBytes, 0, nameBytes.length);
    return new String (nameBytes);
  }

  /**
   * Replaces the pure key data within the specified key bundle and considers 
   * the CRC-32 value.
   * 
   * @param key a given key bundle.
   * @param pureKey the pure key bundle data to be replaced.
   */
  public static void replacePureKey(byte [] key, byte[] pureKey)
  {
    int stInx = KeyFile.getKeyStartingPoint(key);
    System.arraycopy(pureKey, 0, key, stInx, pureKey.length);
    long crc = CRC.getCRC_32(key, 4);
    HexTool.insertIntBigEndianAt(crc, key.length - 4, key);
  }

  /**
   * Extracts the pure key data without header from the given key.
   * 
   * @param key a given key.
   * @return the pure key data without header.
   */
  public static byte[] getPureKeyData(byte[] key)
  {
    byte [] amountBytes = new byte [2]; 
    System.arraycopy(key, HEADER_LENGTH, amountBytes, 0, amountBytes.length);
    byte nameLength = HexTool.parseHex(new String(amountBytes));
    int hLength = nameLength + HEADER_LENGTH_PLUS_NAME_AMOUNT;
    byte [] pureKey = new byte [key.length - hLength - CRC_LENGTH];
    System.arraycopy(key, hLength, pureKey, 0, pureKey.length);
    return pureKey;
  }
  
  /**
   * Extracts the pure key data without header from the key of this KeyFile 
   * instance.
   * 
   * @return the pure key data without header.
   * @throws Exception in case of any error.
   */
  public byte [] getPureKeyData() throws Exception
  {
    byte [] thisKey = getKey();
    return KeyFile.getPureKeyData(thisKey);
  }
  
  /**
   * Calculates the length of pure key data. This is key length minus header 
   * length;
   * @return the length of pure key data.
   * @throws Exception in case of any error.
   */
  public int getPureKeyDataLength() throws Exception
  {
    if (pureKeyDataLength == null)
    {
      pureKeyDataLength = getPureKeyData().length;
    }
    return pureKeyDataLength;
  }
  
  /**
   * Provides a specified part from the given key starting at the specified 
   * position.
   * @param key a given key.
   * @param start the specified starting position.
   * @param length the length of the requested part.
   * @return the specified part from the given key.
   */
  public static byte [] provideKeyPart(byte [] key, int start, int length)
  {
    byte [] pure = getPureKeyData(key);
    byte [] part = new byte [length];
    int pos = start;
    for (int i = 0; i < length; i++)
    {
      part[i] = pure[pos];
      pos++;
      if (pos == pure.length)
      {
        pos = 0;
      }
    }
    return part;
  }
  /**
   * Determines the starting point of the key values. It can be different 
   * because the name of the key is part of the header and its length can vary.
   * 
   * @param key a given key.
   * @return the starting point of the key values.
   */
  public static int getKeyStartingPoint(byte [] key)
  {
    byte [] amountBytes = new byte [2]; 
    System.arraycopy(key, HEADER_LENGTH, amountBytes, 0, amountBytes.length);
    byte nameLength = HexTool.parseHex(new String(amountBytes));
    return HEADER_LENGTH_PLUS_NAME_AMOUNT + nameLength;
  }
  
  /**
   * Returns the associated key. If the key does not exist yet, it will now be 
   * tried to read it from the specified location.
   * 
   * @return the associated key.
   * @throws Exception incase of any file error.
   */
  public byte [] getKey() throws Exception
  {
    if (key == null)
    {
      File f = new File (getPath());
      key = new byte [(int)f.length()];
      try (BufferedInputStream in = new BufferedInputStream(
              new FileInputStream(f))) 
      {
        in.read(key);
      }
    }
    return key;
  }
  
  /**
   * Returns <b>true</b>, if this object contains already a key. It is possible 
   * that this key file object initially contains only information about a key 
   * but not yet the key itself. If the key itself is not yet contained, an 
   * attempt will be made to load it when it is accessed for the first time.
   * 
   * @return <b>true</b>, if this object contains already a key. 
   */
  public boolean hasKey()
  {
    return key != null;
  }
  
  /**
   * Creates a random file name with a length as specified. The name is formed 
   * from a string of randomly generated hexadecimal digits.
   * 
   * @param length the requested length of the name.
   * @return a random file name.
   */
  public static String randomFileName(int length)
  {
    if (random == null)
    {
      random = new Random(System.currentTimeMillis());
    }
    String s = "";
    int c;
    for (int i = 0; i < length; i++)
    {
      c = random.nextInt(15);
      s = s + (char)(c >= 10 ? c + 55 : c + 48);
    }
    return s.toLowerCase();
  }

  /**
    * Return the drive letter of the medium on which the key is located.
   * @return the drive letter of the medium on which the key is located.
   */
  public String getDrive()
  {
    return driveLetter;
  }
  /**
   * Calculates an entry (starting) point into the key by using the given random 
   * factor [0 .. 1[ applied to the length of the key. Since the random factor 
   * is updated only every 100ms and encryption of multiple files is possible 
   * within a much shorter period of time, an additional part is added to the 
   * entry point calculated from random factor times key length and then a new 
   * additional part is obtained from the key values by using the two bytes at 
   * the position of the calculated entry point as an integer value.
   * 
   * @param f a given random double factor [0.0 .. 1.0[.
   * @param key a given key for that an entry point is requested.
   * @return an entry (starting) point into the key.
   */
  private static int getEntryPoint(double f, byte [] key)
  {
    int n = key.length;
    int entryPoint = (int) ((n - 1) * f) + 
            lastEntryPointAdd + 
            HexTool.byteToInt((byte)System.nanoTime());
    
    if (entryPoint > n - 1)
    {
      entryPoint = entryPoint % n;
      entryPoint = entryPoint == (n - 1) ? entryPoint - 1 : entryPoint;
    }
    
    lastEntryPointAdd = HexTool.byteToWordLittleEndian(entryPoint, key);
    return entryPoint;
  }
  
  /**
   * Calculates a desired number of random key entry points and makes sure they
   * are all different. If more than 5% of the possible key entry points are 
   * requested, the entry points are created as an incremental sequence starting 
   * at the point determined by the random factor. This procedure is done because 
   * the time for generation increases too much with a larger request. However, 
   * this has not unavoidable disadvantages, since the bytes of the key are real 
   * random numbers.<br><br>
   * The advantage of getting entry points via this method is that when an 
   * entire packet of files is encrypted, they are actually all encrypted with 
   * a different key.
   * 
   * @param f a given random double factor [0.0 .. 1.0[.
   * @param key the pure key data of a given key.
   * @param count the number of entry points requested.
   * @param existing an array list of already used and forbidden entry points.
   * @return an array list with different integer entry points.
   * @throws Exception in case of more entry point are requested as the key is 
   * able to provide or the random factor is out of range.
   */
  public static ArrayList < Integer > getDifferentEntryPoints (
          double f, 
          byte [] key, 
          int count, 
          ArrayList < Integer > existing) throws Exception
  {
    if (key.length - (existing != null ? existing.size() : 0) < count)
    {
      throw new Exception ("Not enough entry points available.");
    }
    
    if (f < 0 || f >= 1)
    {
      throw new Exception ("Random factor out of range.");
    }
    
    boolean linear = count / (double)key.length > 0.05;
    ArrayList < Integer > entryPoints = new ArrayList <> ();
    int ep = getEntryPoint(f, key);
    while (entryPoints.size() < count)
    {
      ep = linear ? ep + 1 : getEntryPoint(f, key);
      ep = ep > key.length ? 0: ep;
      if ((linear || !entryPoints.contains(ep)) && 
          ((existing != null && !existing.contains(ep)) || existing == null))
      {
        entryPoints.add(ep);
      }
    }
    return entryPoints;
  }
  
  /**
   * Calculates a desired number of random key entry points and makes sure they
   * are all different. This is the right approach if the key is used to encrypt 
   * many files. This ensures that a different key from the keyring was used for 
   * each file. If a potential attacker who gets hold of the encrypted data 
   * tries to decrypt the files in this bundle, he has not had a chance.
   * 
   * @param f a given random double factor [0.0 .. 1.0[.
   * @param key a given key.
   * @param count the number of entry points requested.
   * @return an array list with different integer entry points.
   * @throws Exception in case of more entry point are requested as the key is 
   * able to provide.
   */
  public static ArrayList < Integer > getDifferentEntryPoints (
          double f, 
          byte [] key, 
          int count) throws Exception
  {
    return getDifferentEntryPoints(f, key, count, null);
  }
  
  /**
   * Assumes that the specified file contains a valid key and attempts to read 
   * it to create an associated KeyFile instance. 
   * 
   * @param f a given file.
   * @return a KeyFile instance with the key from the given file or <b>null</b>
   * if there is any problem with the file content.
   */
  public static KeyFile takeAsKey (File f)
  {
    KeyFile keyFile = null;
    byte [] buffer = new byte [Key.KEY_HEADER_LENGTH];
    try
    {
      BufferedInputStream bis = new BufferedInputStream (new FileInputStream(f));
      bis.read(buffer);
      bis.close();
      Long fileId = KeyFile.getKeyFileIdentifier(buffer);
      if (fileId != null)
      {
        int length = (int)f.length();
        byte [] key = new byte [length];
        bis = new BufferedInputStream (new FileInputStream(f));
        bis.read(key);
        bis.close();
        if (KeyFile.isValidKey(key))
        {
          keyFile = new KeyFile(fileId, f.getAbsolutePath(), length, key);
        }
      }
    } 
    catch (Exception e)
    {
      // keyFile remains "null" in case of any error.
    }
    return keyFile;
  }

  /**
   * Sets the number of already used virtual keys. This value must be set by 
   * an external key manager and, if present, is used only for the toString 
   * method.
   * 
   * @param used the number of already used virtual keys.
   */
  public void setUsedKeys(Integer used) 
  {
    this.used = used;
  }

  public static String insertThousandsSeparators(String str) 
  {
    String cstr = "";
    int count = 0;
    for (int i = str.length() - 1; i >= 0; i--)
    {
      count++;
      cstr = str.charAt(i) + cstr;
      if (count % 3 == 0 && i > 0)
      {
        cstr = "." + cstr;
      }
    }
    return cstr;
  }
  
  /**
   * Performs an analysis of how frequently each byte occurs in the given key 
   * bundle and swaps out more frequently occurring bytes for less frequently 
   * occurring bytes until the scatter around the mean is minimized or 
   * approaches zero.
   * 
   * @param orgKey a key bundle to be optimized.
   * @param scat the requested scattering factor.
   * @return a key bundle whose uniform distribution is optimized.
   */
  public static byte [] harmonicallyBalance (byte [] orgKey, double scat)
  {
    byte [] keyBundle = orgKey.clone();
    byte [] bundle = KeyFile.getPureKeyData(keyBundle);
    KeyQualityAssessor kqa = new KeyQualityAssessor();
    Double [] quality = null;
    int [] frequencies = new int [256];
    int [] minMaxIndexes = new int [2];
    boolean cont;
    do
    {
      determineFrequencies(bundle, frequencies);
      determineMinMaxIndexes (frequencies, minMaxIndexes);
      cont = exchange (minMaxIndexes, frequencies, bundle);
      KeyFile.replacePureKey (keyBundle, bundle);
      quality = kqa.determineQuality(keyBundle);
    }
    while (cont && quality[KeyQualityAssessor.SCATTERING] < scat);
    return keyBundle;
  }

  private static void determineFrequencies(byte[] bundle, int [] frequencies) 
  {
    int inx;
    for (int i = 0; i < frequencies.length; i++)
    {
      frequencies[i] = 0;
    }
    for (int i = 0; i < bundle.length; i++)
    {
      inx = HexTool.byteToInt(bundle[i]);
      frequencies[inx] = frequencies[inx] + 1;
    }
  }

  private static void determineMinMaxIndexes(int[] frequencies, int[] minMaxIndexes) 
  {
    int min = Integer.MAX_VALUE;
    int max = 0;
    for (int i = 1; i < frequencies.length; i++)
    {
      if (frequencies[i] < min)
      {
        min = frequencies[i];
        minMaxIndexes[0] = i;
      }
      if (frequencies[i] > max)
      {
        max = frequencies[i];
        minMaxIndexes[1] = i;
      }
    }
  }

  /**
   * Exchange half of the difference between the both frequencies at the given 
   * indexes of the byte with lower frequency against byte with the higher 
   * frequency.
   * 
   * @param minMaxIndexes the both indexes into the frequencies list.
   * @param frequencies the frequecies of byte occurances within the bundle.
   * @param bundle the key bundle to be harmonized.
   * @return <b>true</b> if one or more bytes have been exchanged.
   */
  private static boolean exchange(int[] minMaxIndexes, int[] frequencies, byte[] bundle) 
  {
    boolean exchanged = false;
    Random random = new Random(System.nanoTime());
    int amount = (frequencies[minMaxIndexes[1]] - frequencies[minMaxIndexes[0]]) / 2;
    byte high = (byte) minMaxIndexes[1];
    byte low = (byte) minMaxIndexes[0];
    int inx;
    for (int j = 0; j < amount; j++)
    {
      // Pick an index somewhere randomly within the bundle and increment it 
      // until it reaches a lowbyte and replace it with a highbyte.
      inx = (int) (random.nextDouble(1.0) * bundle.length);
      while (bundle[inx] != high)
      {
        inx++;
        inx = inx == bundle.length ? 0 : inx;
      }
      bundle[inx] = low;
      exchanged = true;
    }
    return exchanged;
  }
  
  /**
   * Calculates quantities that make statements about the difference between two 
   * key bundless. There are three values: The first value indicates how often 
   * the same byte values were found at the same positions, the second value 
   * indicates how often this is not the case, and the third value calculates 
   * the average difference of all byte values at the same positions.  
   * 
   * @param pData_1 one key bundle.
   * @param pData_2 another key bundle.
   * @return the calculated values.
   */
  public static Double [] calculateDiversity (byte [] pData_1, byte [] pData_2)
  {
    int equCount = 0;
    int dists = 0;
    for (int i = 0; i < pData_1.length; i++)
    {
      if (pData_1[i] == pData_2[i])
      {
        equCount++;
      }
      dists = dists + Math.abs(HexTool.byteToInt(pData_1[i]) - HexTool.byteToInt(pData_2[i]));
    }
    double aveDist = dists / (double)pData_1.length;

    Double [] results = new Double [3];
    results[0] = (double)equCount;
    results[1] = (1 - equCount / (double)pData_1.length) * 100;
    results[2] = aveDist;
    return results;
  }
}
