package de.das.encrypter.model;

import java.io.File;

import java.util.ArrayList;

/**
 * Interface for classes which want to use the key files searcher. The 
 * concurrently running autonomous thread (searcher) communicates its results to 
 * its users through this interface.<br><br>
 * How to use the key files searcher:<br><br>
 * The KeyFilesSeacher is activated by invoking 
 * "KeyFilesSearcher.createInstance(receiver)" where "receiver" must be the 
 * instance of a class which implements this "KeyFilesSearchReceiver" interface.
 * <br><br>
 * This receiver class acts as a manager of keys. To do this, it must create an 
 * instance of the "KeyFiles" class. This will be a list of all currently 
 * available keys. If the "KeyFilesSearcher" has performed a search, it will also 
 * pass an instance of the "KeyFiles" class to the receiver class via the method
 * "synchronizeKeyFilesList" with the keys it has found, so that the receiver 
 * can synchronize its key list.
 * <br><br>
 * 
 * @author Frank Keldenich
 */
public interface KeyFilesSearchReceiver 
{
  /**
   * Passes a hash map with found key files to the key file search receiver for
   * it can synchronize its own hash map with key files.
   * 
   * @param files a hash map with found key files.
   */
  public void synchronizeKeyFilesList (KeyFiles files);
  
  /**
   * Request the hash map with key files from the key file search receiver. 
   * 
   * @return the hash map with key files.
   */
  public KeyFiles getKeyFiles();
  
  /**
   * Passes a list of key file locations objects to the key file search 
   * receiver.
   * 
   * @param keyFileLocations an array list with key file locations objects. 
   */
  public void setKeyFileLocations(ArrayList < KeyFileLocation > keyFileLocations);
  
  /**
   * Informs the key file search receiver that the key files searcher is 
   * currently searching for keys.
   * 
   * @param b <b>true</b>, if the key files searcher is currently searching for 
   * keys.
   */
  public void setKeyFileSearching(boolean b);
  
  /**
   * Passes a list of file paths of the currently existing key bundle folders to 
   * the key file search receiver.
   * 
   * @param bundleFolders an array list with paths to existing bundle folders. 
   */
  public void setBundleFolders(ArrayList < File > bundleFolders);
  
  /**
   * Tells the key file searcher that searching for keys is allowed. 
   * @return <b>true</b>, if searching for keys is allowed. 
   */
  public boolean isSearchingAllowed();
}
