package de.das.encrypter.processors;

import de.das.encrypter.model.KeyFile;

/**
 * An assembler is a class that provides the functionality to reassemble data 
 * fragments of a key that has been split by a splitter into one key again. It 
 * is expected that the fragmented key is also encrypted with another key known 
 * to the recipient. After receiving and assembling all fragmented parts, the 
 * first thing to do is to call the method for checking whether the received key 
 * is a valid key. This decrypts thereby also the received data (if the proper 
 * key is available), which can be used then as new key.<br><br>
 * Another requirement for using this class is that the EncoderDecoder has 
 * previously been passed the list of available keys.
 * 
 * @author Frank Keldenich
 */
public class Assembler 
{
  private final int expectedSize;
  
  private final ProgressListener progressListener;
  
  private int bufferPointer = 0;
  
  private final byte [] data;
  
  private byte [] key = null;

  /**
   * Creates a new data assembler for assembling a fragmented key. The 
   * prerequisite is that the fragments arrive at the assembler in the correct 
   * order. 
   * 
   * @param expectedSize the expected size of the whole data.
   * @param pl an instance implementing the interface ProgressListener or 
   * <b>null</b> if no progress listener is used.
   */
  public Assembler(int expectedSize, ProgressListener pl)
  {
    this.expectedSize = expectedSize;
    this.progressListener = pl;
    data = new byte [expectedSize];
  }
  /**
   * Writes the received data fragment to the collection buffer.
   * 
   * @param fragment the next part of the expected data.
   */
  public void setData(byte[] fragment) 
  {
    System.arraycopy(fragment, 0, data, bufferPointer, fragment.length);
    bufferPointer = bufferPointer + fragment.length;
    if (progressListener != null)
    {
      progressListener.setCurrentAmount(bufferPointer);
    }
  }

  /**
   * Checks if the received data is decryptable and after decryption if it is a 
   * valid key. The check includes verification of the correct data length and 
   * whether the required key for decryption is available. If the check is 
   * successful, the encrypted key is also available.
   * 
   * @return <b>true</b>, if the received data is a valid decryptable key.
   */
  public boolean isValidKey() 
  {
    boolean valid = data.length == expectedSize;
    if (valid)
    {
      EncoderDecoder ed = new EncoderDecoder();
      try 
      {
        key = ed.decrypt(data);
        valid = KeyFile.isValidKey(key);
      } 
      catch (Exception e) 
      {
        key =null;
        valid = false;
      }
    }
    return valid;
  }

  /**
   * Provides the received and decrypted key. It is available if the validation 
   * check was successful.
   * 
   * @return the received and decrypted key or <b>null</b>, if the key could not 
   * be assembled or decrypted.
   */
  public byte[] getKey() 
  {
    return key;
  }

  /**
   * Return the received data independent on it is a key or any other data.
   * @return the received data independent on it is a key or any other data.
   */
  public byte[] getData() 
  {
    return data;
  }
  
  
}
