package de.das.encrypter.processors;

import de.das.encrypter.tools.CRC;
import de.das.encrypter.tools.HexTool;

import static java.lang.Thread.sleep;

import javax.swing.Timer;

/**
 * The task of the block sender is to send a given byte array in portions via 
 * UDP to a receiver. The division into smaller blocks is necessary for large 
 * byte arrays because the data sent via UDP is limited in size.
 * 
 * @author Frank Keldenich
 */
public class BlockSender extends Thread
  {
    private final byte [] frame = new byte [
            KeyTransfer.BUFFER_SIZE + 
            KeyTransfer.ADD_DATA_SIZE];
    
    private byte [] block;
    
    private Splitter splitter;
    
    private final byte [] payload;
    
    private final TransferControl ctrl;
    
    private final Timer connectionTimeOut;
    
    public BlockSender(
            byte [] payload, 
            ProgressListener pl, 
            TransferControl ctrl, 
            Timer timer)
    {
      this.payload = payload;
      this.ctrl = ctrl;
      this.connectionTimeOut = timer;
      init(pl);
    }

    private void init(ProgressListener pl)
    {
      splitter = new Splitter(payload, pl);
    }
    
    @Override
    public void run()
    {
      ctrl.setFrameId(splitter.getBlockCount());
      block = splitter.getNext();
      ctrl.setNextFrame(true);
      while (block != null && !ctrl.isBreakSender())
      {
        if (ctrl.isNextFrame())
        {
          buildFrame (ctrl.getFrameId(), payload.length, block);
        }
        ctrl.setWaitForResponse(true);
        ctrl.setResponse(false);
        ctrl.setNextFrame(false);
        ctrl.getSender().callServer(frame);
        connectionTimeOut.restart();
        waitForResponse();
        if (ctrl.isNextFrame())
        {
          ctrl.setFrameId(splitter.getBlockCount());
          block = splitter.getNext();
        }
      }
      if (block == null && !ctrl.isBreakSender())
      {
        ctrl.setWaitForResponse(true);
        ctrl.setResponse(false);
        ctrl.getSender().callServer(KeyTransfer.EOT);
        connectionTimeOut.restart();
        waitForResponse();
      }
    }
    
    private void buildFrame(int id, int keySize, byte [] block)
    {
      System.arraycopy(block, 0, frame, KeyTransfer.POS_BLOCK_START, block.length);
      HexTool.insertIntBigEndianAt(id, KeyTransfer.POS_BLOCK_COUNT, frame);
      HexTool.insertIntBigEndianAt(keySize, KeyTransfer.POS_KEY_LENGTH, frame);
      HexTool.insertIntBigEndianAt(block.length, KeyTransfer.POS_DATA_SIZE, frame);
      int crc = (int) CRC.getCRC_32(frame, 4);
      HexTool.insertIntBigEndianAt(crc, frame.length - 4, frame);
    }

    @SuppressWarnings({"SleepWhileInLoop"})
    private void waitForResponse()
    {
      while (!ctrl.isResponse() && !ctrl.isBreakSender())
      {
        try
        {
          sleep(50);
        } 
        catch (InterruptedException e)
        {
        }
      }
    }
  }
