package de.das.encrypter.processors;

import de.das.encrypter.model.InvalidKeyException;

import de.das.encrypter.tools.CRC;
import de.das.encrypter.tools.HexTool;

import de.das.encrypter.model.Key;
import de.das.encrypter.model.KeyFile;
import de.das.encrypter.model.KeyFiles;
import de.das.encrypter.model.NoKeyAvailableException;
import de.das.encrypter.model.NotEncryptedException;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.RandomAccessFile;

import java.util.Arrays;

/**
 * This class provides methods for encryption and decryption using the unique 
 * and random keys created by this factory.
 * 
 * @author Dipl. Phys.-Ing. Frank Keldenich
 */
public class EncoderDecoder 
{
  private static String noFolderName;
  
  private static KeyFiles keyFiles = null;
  
  /**
   * Static method for commissioning the encoder-decoder, which provides the 
   * processor with the keys known to the system. The name substitution for
   * folder names that cannot be decrypted is set to "".
   * 
   * @param kfs the instance of the hash map with the known key files.
   */
  public static void setup(KeyFiles kfs)
  {
    setup ("", kfs);
  }
 
  /**
   * Static method for commissioning the encoder-decoder, which provides the 
   * processor with the keys known to the system.
   * 
   * @param name a string or character used when trying to decrypt encrypted 
   * folder names and the decryption fails (e.g. the character ). 
   * @param kfs the instance of the hash map with the known key files.
   */
  public static void setup(String name, KeyFiles kfs)
  {
    noFolderName = name;
    keyFiles = kfs;
  }
 
  /**
   * Determines the original folder name of the given folder with an cryptic 
   * name.
   * 
   * @param f an folder with an encrypted name.
   * @return the original unencrypted folder name.
   */
  public static String getFolderName(File f)
  {
    return getFolderName (f.getAbsolutePath(), f.getName());
  }
    
  /**
   * Determines the original folder name of the the given path with the cryptic 
   * name.
   * 
   * @param path the path to a folder with an encrypted name.
   * @param lastFolder the file name of the file containing the encrypted original 
   * folder name.
   * @return the original unencrypted folder name.
   */
  public static String getFolderName(String path, String lastFolder)
  {
    String folderName = " Error ";
    path = path.substring(0, path.indexOf(lastFolder) + lastFolder.length()) + File.separator + lastFolder;
//    path = path + File.separator + lastFolder;
    File nameContainerFile = new File (path);
    int len = (int) nameContainerFile.length();
    byte [] folderFileBytes = new byte [len];
    try
    {
      try (BufferedInputStream in = new BufferedInputStream(
              new FileInputStream(nameContainerFile))) 
      {
        in.read(folderFileBytes);
      }
      
      byte [] buffer = new byte [16];
      System.arraycopy(folderFileBytes, 0, 
          buffer, 0, buffer.length);
      long keyId = Long.parseLong(new String(buffer), 16);
      KeyFile kf = keyFiles.get(keyId);
      if (kf != null)
      {
        byte [] key = kf.getKey();
        byte [] nameBytes = new byte [folderFileBytes.length - 16];
        System.arraycopy(folderFileBytes, 16, nameBytes, 0, nameBytes.length);
        for (int i = 0; i < nameBytes.length; i++)
        {
          nameBytes[i] = (byte) (nameBytes[i] ^ key[i]);
        }
        folderName = new String (nameBytes);
      }
      else
      {
        folderName = noFolderName;
      }
    } 
    catch (Exception e)
    {
    }
    if (folderName.contains("Error"))
    {
      folderName = folderName;
    }
    return folderName;
  }
  
  /**
   * Extracts the key ID from an encrypted file.
   * 
   * @param f a file with encrypted data.
   * @return the ID of the required key for the decryption.
   * @throws NotEncryptedException if the given data are not encrypted.
   * @throws Exception in case of any error or if the file does not contain 
   * encrypted data.
   */
  public long getKeyId (File f) throws Exception, NotEncryptedException
  {
    byte [] content = new byte [Key.ENCRYPTION_PREFIX_LENGHT];
    try (BufferedInputStream bis = new BufferedInputStream(new FileInputStream(f))) 
    {
      bis.read(content);
    }
    return getKeyId(content);
  }
  
  /**
   * Extracts the key ID from an byte array containing encrypted data.
   * 
   * @param content a byte array of encrypted data.
   * @return the ID of the key required for the decryption of the given 
   * encrypted data.
   * @throws NotEncryptedException if the given data are not encrypted.
   * @throws Exception in case of any other error.
   */
  public static long getKeyId (byte [] content) throws Exception, NotEncryptedException
  {
    if (isEncrypted(content))
    {
      byte [] buffer = new byte [16];
      System.arraycopy(content, Key.ENCRYPTION_PREFIX_KEY_ID_POSITION, 
              buffer, 0, buffer.length);
      long keyId = Long.parseLong(new String(buffer), 16);
      return keyId;
    }
    else
    {
      throw new NotEncryptedException ();
    }
  }
  
  /**
   * Provides the random entry point into the key.
   * 
   * @param f a file with encrypted data.
   * @return the entry point into the encryption key.
   * @throws Exception in case of any error.
   */
  public static long getKeyEntryPoint (File f) throws Exception
  {
    byte [] buffer = new byte [48];
    try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(f)))
    {
      in.read(buffer);
    }
    return getKeyEntryPoint(buffer);
  }
  
  /**
   * Provides the random entry point into the key.
   * 
   * @param content a byte array of encrypted data.
   * @return the entry point into the encryption key.
   * @throws Exception in case of any error.
   */
  public static long getKeyEntryPoint (byte [] content) throws Exception
  {
    byte [] buffer = new byte [16];
    
    System.arraycopy(content, Key.ENCRYPTION_PREFIX_ENTRY_POINT_MSB_POSITION, 
            buffer, 0, buffer.length);
    String eStr = new String (buffer);
    while (eStr.startsWith("0") && eStr.length() > 1)
    {
      eStr = eStr.substring(1);
    }
    long entry = Long.parseLong(eStr, 16);
    return entry;
  }
  
  /**
   * Assumes that the specified byte array is encrypted and attempts to decrypt 
   * the data using the key whose ID is in the header of the encrypted data.
   * 
   * @param data a byte array with encrypted data.
   * @return the decrypted data.
   * 
   * @throws NoKeyAvailableException if no key with the ID contained in the 
   * header of the encrypted data is available.
   * @throws Exception if any other error occurs.
   */
  public byte [] decrypt (byte [] data) throws NoKeyAvailableException, Exception
  {
    byte [] dcontent = null;
    long keyId = getKeyId(data);
    // Get the associated key.
    KeyFile keyFile = keyFiles.get(keyId);
    if (keyFile != null)
    {
      byte [] key = keyFile.getKey();
      dcontent = decrypt(key, data);
    }
    else
    {
      throw new NoKeyAvailableException();
    }
    return dcontent;
  }
  
  /**
   * Assumes that the given byte array is encrypted with the given key and 
   * tries to decrypt this data.
   * 
   * @param key a given key.
   * @param data a byte array with encrypted data.
   * @return the decrypted data.
   * @throws Exception in case of a failed decryption.
   */
  public byte [] decrypt (byte [] key, byte [] data) throws Exception
  {
    if (data.length < Key.ENCRYPTION_PREFIX_LENGHT)
    {
      throw new Exception ("File length too less.");
    }
    // Get the pure key data.
    byte [] pureKey = KeyFile.getPureKeyData(key);
    // Get the entry point from the source.
    long entry = getKeyEntryPoint(data);
    // Get the key name to use its length.
    byte [] nameBytes = getEncryptedFileName(data);
    // Free the encrypted content from the leading header.
    byte [] content = new byte [data.length - (Key.ENCRYPTION_PREFIX_LENGHT + 2 + nameBytes.length)];
    System.arraycopy(data, Key.ENCRYPTION_PREFIX_LENGHT + 2 + nameBytes.length, content, 0, content.length);

    int pointer = (int) entry;
    for (int inx = 0; inx < content.length; inx++)
    {
      if (pointer >= pureKey.length)
      {
        pointer = 0;
      }
      content[inx] = (byte) (content[inx] ^ pureKey[pointer]);
      pointer++;
    }
    return content;
  }
  
  /**
   * Encrypts the specified data using the specified key, with key usage 
   * starting at the specified entry position. The encrypted data is preceded by
   * a header, by which it can be recognized that it is encrypted data, with 
   * which key (key ID) it was encrypted, from which position the bytes of the 
   * key are to be used and what the file with the unencrypted data was called.
   * 
   * @param entryPoint the position where to start using byte in the key.
   * @param key a valid encryption key.
   * @param data the data to be encrypted.
   * <b>null</b> if not used.
   * @return a byte array with the encrypted data.
   * @throws InvalidKeyException if the key is invalid.
   * @throws Exception in all other error cases.
   */
  public byte [] encrypt (
          long entryPoint, 
          byte [] key, 
          byte [] data) throws InvalidKeyException, Exception
  {
    return encrypt(entryPoint, key, data, null);
  }
  
  /**
   * Encrypts the specified data using the specified key, with key usage 
   * starting at the specified entry position. The encrypted data is preceded by
   * a header, by which it can be recognized that it is encrypted data, with 
   * which key (key ID) it was encrypted, from which position the bytes of the 
   * key are to be used and what the file with the unencrypted data was called.
   * 
   * @param entryPoint the position where to start using byte in the key.
   * @param key a valid encryption key.
   * @param data the data to be encrypted.
   * @param fileName the name of the the file where the data come from or 
   * <b>null</b> if not used.
   * @return a byte array with the encrypted data.
   * @throws InvalidKeyException if the key is invalid.
   * @throws Exception in all other error cases.
   */
  public byte [] encrypt (
          long entryPoint, 
          byte [] key, 
          byte [] data, 
          String fileName) throws InvalidKeyException, Exception
  {
    byte [] pureKey = KeyFile.getPureKeyData(key);
    byte [] nameBytes = fileName == null ? new byte [0] : fileName.getBytes();
    int upPointer = (int)entryPoint;
    for (int inx = 0; inx < nameBytes.length; inx++)
    {
      if (upPointer >= pureKey.length)
      {
        upPointer = 0;
      }
      nameBytes[inx] = (byte) (nameBytes[inx] ^ pureKey[upPointer]);
      upPointer++;
    }
    
    byte [] signatureBytes = new byte [Key.KEY_SIGNATURE_PREFIX.length()];
    System.arraycopy(key, 0, signatureBytes, 0, signatureBytes.length);
    String signature = new String (signatureBytes);
    if (!signature.equals(Key.KEY_SIGNATURE_PREFIX))
    {
      throw new InvalidKeyException ();
    }
    
    byte [] encryptionFixPart = new byte [Key.ENCRYPTION_PREFIX_LENGHT];
    byte [] encryptionPrefix = new byte [Key.ENCRYPTION_PREFIX_LENGHT + nameBytes.length + 2];
    System.arraycopy(Key.ENCRYPTION_ID.getBytes(), 0, 
            encryptionFixPart, Key.ENCRYPTION_PREFIX_ID_POSITION, 
            Key.ENCRYPTION_ID_LENGTH);
    System.arraycopy(key, Key.KEY_SIGNATURE_PREFIX.length(), 
            encryptionFixPart, Key.ENCRYPTION_PREFIX_KEY_ID_POSITION, 
            Key.KEY_ID_LENGHT);
    System.arraycopy(HexTool.toTwoHexQuad((int)((entryPoint >> 32) & 0xFFFFFFFF)).getBytes(), 0, 
            encryptionFixPart, Key.ENCRYPTION_PREFIX_ENTRY_POINT_MSB_POSITION, 
            8);
    System.arraycopy(HexTool.toTwoHexQuad((int)(entryPoint & 0xFFFFFFFF)).getBytes(), 0, 
            encryptionFixPart, Key.ENCRYPTION_PREFIX_ENTRY_POINT_LSB_POSITION, 
            8);
    
    int crc = CRC.getCRC_16(encryptionFixPart, 2);
    HexTool.insertShortIntLittleEndianAt(
            crc, 
            Key.ENCRYPTION_PREFIX_CRC_16_POSITION,
            encryptionFixPart);
    System.arraycopy(encryptionFixPart, 0, 
            encryptionPrefix, 0, 
            encryptionFixPart.length);
    HexTool.insertShortIntLittleEndianAt(
            nameBytes.length, 
            Key.ENCRYPTION_PREFIX_NAME_LENGTH_POSITION, 
            encryptionPrefix);
    System.arraycopy(nameBytes, 0, 
            encryptionPrefix, Key.ENCRYPTION_PREFIX_FILE_NAME_POSITION, 
            nameBytes.length);
    
    byte [] econtent = new byte [data.length + encryptionPrefix.length];
    System.arraycopy(encryptionPrefix, 0, econtent, 0, encryptionPrefix.length);
    
    int offset = encryptionPrefix.length;
    try
    {
      upPointer = (int)entryPoint;
      for (int inx = 0; inx < data.length; inx++)
      {
        if (upPointer >= pureKey.length)
        {
          upPointer = 0;
        }
        econtent[inx + offset] = (byte) (data[inx] ^ pureKey[upPointer]);
        upPointer++;
      }
    } 
    catch (ArrayIndexOutOfBoundsException e)
    {
    }
    return econtent;
  }
  
  /**
   * This encryption method assumes that the given key is a part of the key 
   * identified by the given ID started at the given entry point position. So
   * it encrypts the given data with the pure key data and inserts key ID and
   * entry point in the encryption header.
   * 
   * @param keyId
   * @param pureKey
   * @param entryPoint
   * @param data
   * @return
   * @throws InvalidKeyException
   * @throws Exception 
   */
  public byte [] encrypt (
          long keyId,
          byte [] pureKey, 
          long entryPoint, 
          byte [] data,
          String fileName) throws InvalidKeyException, Exception
  {
    byte [] nameBytes = fileName == null ? new byte [0] : fileName.getBytes();
    int upPointer = (int)entryPoint;
    for (int inx = 0; inx < nameBytes.length; inx++)
    {
      if (upPointer >= pureKey.length)
      {
        upPointer = 0;
      }
      nameBytes[inx] = (byte) (nameBytes[inx] ^ pureKey[upPointer]);
      upPointer++;
    }

    byte [] encryptionFixPart = new byte [Key.ENCRYPTION_PREFIX_LENGHT];
    byte [] encryptionPrefix = new byte [Key.ENCRYPTION_PREFIX_LENGHT + 2 + nameBytes.length];
    System.arraycopy(Key.ENCRYPTION_ID.getBytes(), 0, 
            encryptionFixPart, Key.ENCRYPTION_PREFIX_ID_POSITION, 
            Key.ENCRYPTION_ID_LENGTH);
    
    String hexId = 
            HexTool.toTwoHexQuad((int)((keyId >> 32) & 0xFFFFFFFF)) +
            HexTool.toTwoHexQuad((int)(keyId & 0xFFFFFFFF));
    System.arraycopy(hexId.getBytes(), 0, 
            encryptionFixPart, Key.ENCRYPTION_PREFIX_KEY_ID_POSITION, 
            Key.KEY_ID_LENGHT);
    
    System.arraycopy(HexTool.toTwoHexQuad((int)((entryPoint >> 32) & 0xFFFFFFFF)).getBytes(), 0, 
            encryptionFixPart, Key.ENCRYPTION_PREFIX_ENTRY_POINT_MSB_POSITION, 
            8);
    System.arraycopy(HexTool.toTwoHexQuad((int)(entryPoint & 0xFFFFFFFF)).getBytes(), 0, 
            encryptionFixPart, Key.ENCRYPTION_PREFIX_ENTRY_POINT_LSB_POSITION, 
            8);
    
    int crc = CRC.getCRC_16(encryptionFixPart, 2);
    HexTool.insertShortIntLittleEndianAt(
            crc, 
            Key.ENCRYPTION_PREFIX_CRC_16_POSITION,
            encryptionFixPart);
    System.arraycopy(encryptionFixPart, 0, 
            encryptionPrefix, 0, 
            encryptionFixPart.length);
    HexTool.insertShortIntLittleEndianAt(
            nameBytes.length, 
            Key.ENCRYPTION_PREFIX_NAME_LENGTH_POSITION, 
            encryptionPrefix);
    System.arraycopy(nameBytes, 0, 
            encryptionPrefix, Key.ENCRYPTION_PREFIX_FILE_NAME_POSITION, 
            nameBytes.length);
    
    byte [] econtent = new byte [data.length + encryptionPrefix.length];
    System.arraycopy(encryptionPrefix, 0, econtent, 0, encryptionPrefix.length);
    
    int offset = encryptionPrefix.length;
    try
    {
      upPointer = 0;
      for (int inx = 0; inx < data.length; inx++)
      {
        if (upPointer >= pureKey.length)
        {
          upPointer = 0;
        }
        econtent[inx + offset] = (byte) (data[inx] ^ pureKey[upPointer]);
        upPointer++;
      }
    } 
    catch (ArrayIndexOutOfBoundsException e)
    {
    }
    return econtent;
  }

  /**
   * Search in the given folder for a file with the key prefix and the given key
   * identifier.
   * 
   * @param folder a given folder.
   * @param keyId the key identifier.
   * @return the key file content or <b>null</b>, if no suitable file could be 
   * found.
   */
  public byte[] getKeyById(String folder, long keyId)
  {
    byte [] key = null;
    byte [] idBytes = new byte [16];
    long checkId;
    File [] files = new File (folder).listFiles();
    for (File f: files)
    {
      if (f.length() > Key.KEY_SIGNATURE_PREFIX.length() + 16)
      {
        try
        {
          byte [] keySignature = new byte [Key.KEY_SIGNATURE_PREFIX.length() + 16];
          BufferedInputStream bis = new BufferedInputStream(new FileInputStream(f));
          bis.read(keySignature);
          bis.close();
          System.arraycopy(keySignature, 8, idBytes, 0, idBytes.length);
          checkId = Long.parseLong(new String (idBytes), 16);
          if (checkId == keyId)
          {
            key = new byte [(int)f.length()];
            bis = new BufferedInputStream(new FileInputStream(f));
            bis.read(key);
            bis.close();
          }
          if (key != null)
          {
            break;
          }
        } 
        catch (Exception e)
        {
          e.printStackTrace();
        }
      }
    }
    return key;
  }

  /**
   * Checks the given buffer content, if it starts with the legal prefix of 
   * an encrypted file.
   * 
   * @param buffer the first 40 bytes of a file. 
   * @return <b>true</b>, if the given buffer content contains the legal prefix 
   * of an encrypted file. 
   */
  public static boolean isEncrypted(byte[] buffer)
  {
    byte [] id = new byte [Key.ENCRYPTION_ID_LENGTH];
    System.arraycopy(buffer, Key.ENCRYPTION_PREFIX_ID_POSITION, id, 0, id.length);
    boolean ok = Arrays.equals(id, Key.ENCRYPTION_ID.getBytes());
    return ok;
  }

  /**
   * Checks the given file content, if it starts with the legal prefix of 
   * an encrypted file.
   * 
   * @param f a given file.
   * @return <b>true</b>, if the given file content starts with the legal prefix 
   * of an encrypted file.
   * @throws Exception in case of any error.
   */
  public boolean isEncrypted(File f) throws Exception
  {
    byte[] buffer = new byte [Key.ENCRYPTION_PREFIX_LENGHT];
    try (BufferedInputStream in = new BufferedInputStream (new FileInputStream (f))) 
    {
      in.read(buffer);
    }
    return isEncrypted(buffer);
  }

  /**
   * Extracts and decrypts the name of the original unencrypted file from the 
   * header of a file with encrypted data.
   * 
   * @param path the path of a file with encrypted data.
   * @return the name of the original unencrypted file.
   */
  public static String getFileName (String path)
  {
    String name = null;
    try
    {
      byte [] first = new byte [Key.ENCRYPTION_PREFIX_LENGHT + 2];
      RandomAccessFile raf = new RandomAccessFile(new File (path), "r");
      raf.read(first);
      raf.close();
      int nameLength = HexTool.byteToShortIntLittleEndian(Key.ENCRYPTION_PREFIX_NAME_LENGTH_POSITION, first);
      byte [] fullPrefix = new byte [Key.ENCRYPTION_PREFIX_LENGHT + 2 + nameLength];

      raf = new RandomAccessFile(new File (path), "r");
      raf.read(fullPrefix);
      raf.close();
      name = getDecryptedFileName(fullPrefix);
    } 
    catch (Exception e)
    {
      name = null;
    }
    return name;
  }
  
  /**
   * Extracts the decrypted name of the original unencrypted file. 
   * 
   * @param econtent the bytes of encrypted data.
   * @return the decrypted name of the original unencrypted file.
   * @throws Exception in case of any error. 
   */
  public static String getDecryptedFileName(byte[] econtent) throws Exception
  {
    byte [] nameBytes = getEncryptedFileName(econtent);
    long entry = getKeyEntryPoint(econtent);
    long keyId = getKeyId(econtent);
    byte [] key = keyFiles.get(keyId).getPureKeyData();
    int pointer = (int) entry;
    for (int inx = 0; inx < nameBytes.length; inx++)
    {
      if (pointer >= key.length)
      {
        pointer = 0;
      }
      nameBytes[inx] = (byte) (nameBytes[inx] ^ key[pointer]);
      pointer++;
    }
    return new String (nameBytes);
  }
  
  /**
   * Extracts the encrypted name of the original unencrypted file from the 
   * header of the given encrypted data.
   * 
   * @param econtent a byte array with encrypted data.
   * @return the encrypted name of the original unencrypted file from the 
   * header of the given encrypted data.
   */
  public static byte[] getEncryptedFileName(byte[] econtent)
  {
    int nameLength = HexTool.byteToShortIntLittleEndian(Key.ENCRYPTION_PREFIX_NAME_LENGTH_POSITION, 
            econtent);
    byte [] nameBytes = new byte[nameLength];
    System.arraycopy(econtent, Key.ENCRYPTION_PREFIX_FILE_NAME_POSITION, 
            nameBytes, 0, 
            nameLength);
    return nameBytes;
  }
  
  /**
   * Extracts the pure encrypted data what means the part of the given data 
   * without the header.
   * @param eData the encrypted data inclusive its header.
   * @return the pure encrypted data without its header.
   */
  public byte [] getPureEncryptedData (byte [] eData) throws Exception
  {
    if (!isEncrypted(eData))
    {
      throw new Exception ("Not encrypted.");
    }
    int nameLength = HexTool.byteToShortIntLittleEndian(Key.ENCRYPTION_PREFIX_NAME_LENGTH_POSITION, 
            eData);
    int pureDataSize = eData.length - (Key.ENCRYPTION_PREFIX_FILE_NAME_POSITION + nameLength);
    byte [] pure = new byte [pureDataSize];
    System.arraycopy(eData, Key.ENCRYPTION_PREFIX_FILE_NAME_POSITION + nameLength, pure, 0, pure.length);
    return pure;
  }
}
