package de.das.encrypter.processors;

import de.das.encrypter.tools.BaseThread;

import de.das.encrypter.model.Key;
import de.das.encrypter.model.KeyFile;
import de.das.encrypter.model.KeyFileLocation;
import de.das.encrypter.model.KeyFiles;
import de.das.encrypter.model.KeyFilesSearchReceiver;

import de.das.encrypter.tools.HiddenDataTool;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;

import java.util.ArrayList;
import java.util.Arrays;

/**
 * The searcher scans periodicaly all drives since letter "C" and looks for
 * folders named "Bundle". Then looks for possible key files contained in the 
 * found folders.
 * 
 * The found key files will be synchronized with the current content of the
 * key files list provided by the search receiver.
 * 
 * @author Frank Keldenich
 */
public class KeyFilesSearcher extends BaseThread
{
  private final static int PERIOD = 10;
  
  private boolean active = false;
  
  private static KeyFilesSearcher searcher;
  
  private final KeyFilesSearchReceiver searchReceiver;
  
  private final HiddenDataTool hdt = new HiddenDataTool();
  
  /**
   * Creates and starts an instance of a key files searcher. Only one searcher 
   * can be used for key files. If an instance has already been created, further 
   * calls to this method will have no effect. This instance runs with the 
   * default period of 10 seconds.
   * 
   * @param searchReceiver instance of a class which implements the interface 
   * KeyFilesSearchReceiver.
   */
  public static void createInstance(KeyFilesSearchReceiver searchReceiver)
  {
    createInstance(searchReceiver, PERIOD);
  }
  
  /**
   * Creates and starts an instance of a key files searcher. Only one searcher 
   * can be used for key files. If an instance has already been created, further 
   * calls to this method will have no effect. 
   * 
   * @param searchReceiver instance of a class which implements the interface 
   * KeyFilesSearchReceiver.
   * @param period the time in seconds between two searchings.
   */
  public static void createInstance(
          KeyFilesSearchReceiver searchReceiver, 
          int period)
  {
    if (searcher == null)
    {
      searcher = new KeyFilesSearcher(searchReceiver, period * 1000);
      searcher.start();
    }
  }

  private KeyFilesSearcher(KeyFilesSearchReceiver searchReceiver, int period)
  {
    this.searchReceiver = searchReceiver;
    setSleepTime(period);
    init();
  }

  private void init()
  {
    setName("Key files searcher");
  }
  
  /**
   * Performs a new search for keys if the search is not currently running and 
   * the key file search receiver allows the search. If a new search starts and 
   * ends, the key file search receiver is notified.
   */
  @Override
  public void doTask()
  {
    if (!active && searchReceiver.isSearchingAllowed())
    {
      active = true;
      search();
      active = false;
    }
  }

  private void search()
  {
    searchReceiver.setKeyFileSearching(true);
    ArrayList < KeyFileLocation > keyFileLocations = new ArrayList <> ();
    KeyFiles files = new KeyFiles();
    ArrayList < File > bundleFolders = searchForBundle();
    searchReceiver.setBundleFolders(bundleFolders);
    Long fileId;
    if (!bundleFolders.isEmpty())
    {
      for (File bundleFolder: bundleFolders)
      {
        File [] content = bundleFolder.listFiles();
        byte [] buffer = new byte [Key.KEY_HEADER_LENGTH];
        if (content != null)
        {
          for (File f: content)
          {
            try
            {
              BufferedInputStream bis = new BufferedInputStream (new FileInputStream(f));
              bis.read(buffer);
              bis.close();
              fileId = KeyFile.getKeyFileIdentifier(buffer);
              if (fileId != null)
              {
                int length = (int)f.length();
                byte [] key = new byte [length];
                bis = new BufferedInputStream (new FileInputStream(f));
                bis.read(key);
                bis.close();
                if (KeyFile.isValidKey(key))
                {
                  files.put(fileId, new KeyFile(fileId, f.getAbsolutePath(), length));
                  keyFileLocations.add (new KeyFileLocation(
                          KeyFile.getKeyName(key), fileId, f.getAbsolutePath()));
                }
              }
            } 
            catch (Exception e)
            {
              e.printStackTrace();
            }
          }
        }
      }
    }
    
    ArrayList < String > secretContainers = searchForSecretContainers();
    if (!secretContainers.isEmpty())
    {
      for (String drive: secretContainers)
      {
        searchForHiddenKeys (drive, new File (drive), files, keyFileLocations);
      }
    }

    updateKeyFilesList (searchReceiver.getKeyFiles());
    searchReceiver.synchronizeKeyFilesList (files);
    searchReceiver.setKeyFileLocations(keyFileLocations);
    searchReceiver.setKeyFileSearching(false);
  }

  /**
   * Scans all drives for a key drive mark what indicates that the drive might
   * containes hidden keys.
   * 
   * @return all folders which are assumed to contain hidden key files or <b>null</b>.
   */
  private ArrayList < String >  searchForSecretContainers()
  {
    ArrayList < String > foundFiles = new ArrayList <> ();
    File f;
    String rootStr;
    char [] root = {'C'};
    
    while (root[0] <= 'Z')
    {
      rootStr = new String (root) + ":\\" + Key.KEY_DRIVE_MARK_FILE_NAME;
      f = new File (rootStr);
      if (f.exists())
      {
        try
        {
          byte [] buffer = new byte [Key.KEY_DRIVE_MARK.getBytes().length];
          try (BufferedInputStream in = new BufferedInputStream(
                                        new FileInputStream(f))) 
          {
            in.read(buffer);
          }
          if (Arrays.equals(buffer, Key.KEY_DRIVE_MARK.getBytes()))
          {
            foundFiles.add (new String (root) + ":\\");
          }
        } 
        catch (Exception e)
        {
          e.printStackTrace();
        }
      }
      root[0] = (char)((int)root[0] + 1);
    }
    return foundFiles;
  }

  /**
   * Scans all drives for a folder named "Bundle" in their root folders.
   * 
   * @return all folders which are assumed to contain key files or <b>null</b>.
   */
  private ArrayList < File >  searchForBundle()
  {
    ArrayList < File > foundFiles = new ArrayList <> ();
    File f;
    String rootStr;
    char [] root = {'C'};
    
    while (root[0] <= 'Z')
    {
      rootStr = new String (root) + ":\\";
      f = new File (rootStr);
      if (f.exists())
      {
        File [] files = f.listFiles();
        if (files != null)
        {
          for (File ff: files)
          {
            if (ff.isDirectory() && ff.getName().equals(KeyFiles.FOLDER_NAME))
            {
              foundFiles.add(ff);
            }
          }
        }
      }
      root[0] = (char)((int)root[0] + 1);
    }
    return foundFiles;
  }

  /**
   * Checks, whether the key files contained in the given list are currently 
   * still available.<br><br>
   * No longer available key files will be removed from the list.
   * 
   * @param keyFiles the list with the detected key files.
   */
  private void updateKeyFilesList(KeyFiles keyFiles)
  {
    KeyFile keyFile;
    for (Long id: keyFiles.keySet().toArray(new Long[0]))
    {
      keyFile = keyFiles.get(id);
      if (keyFile.getPath() != null && !new File(keyFile.getPath()).exists())
      {
        keyFiles.remove(id);
      }
      if (keyFile.getPath() == null && !new File(keyFile.getDrive()).exists())
      {
        keyFiles.remove(id);
      }
    }
  }

  private void searchForHiddenKeys(
          String drive, 
          File root, 
          KeyFiles files, 
          ArrayList < KeyFileLocation > keyFileLocations)
  {
    File [] list = root.listFiles();
    if (list != null)
    {
      for (File f: list)
      {
        if (f.isDirectory())
        {
          searchForHiddenKeys(drive, f, files, keyFileLocations);
        }
        else
        {
          try 
          {
            if (f.getName().toLowerCase().endsWith(".wav") && hdt.keyInWav(f))
            {
              byte [] key = hdt.byteArrayFromWav(f);
              if (KeyFile.isValidKey(key))
              {
                long id = KeyFile.getKeyFileIdentifier(key);
                files.put(id, new KeyFile(id, key, drive));
                keyFileLocations.add (new KeyFileLocation(
                        KeyFile.getKeyName(key), id, f.getAbsolutePath()));
              }
            }
            else if (f.getName().toLowerCase().endsWith(".png") && 
                     (hdt.keyInPng(f) || hdt.keyInQr(f)))
            {
              byte [] key = hdt.byteArrayFromQr(f);
              if (key != null)
              {
                if (KeyFile.isValidKey(key))
                {
                  long id = KeyFile.getKeyFileIdentifier(key);
                  files.put(id, new KeyFile(id, key, drive));
                  keyFileLocations.add (new KeyFileLocation(
                          KeyFile.getKeyName(key), id, f.getAbsolutePath()));
                }
              }
              else
              {
                key = hdt.byteArrayFromImg(f);
                if (key != null)
                {
                  if (KeyFile.isValidKey(key))
                  {
                    long id = KeyFile.getKeyFileIdentifier(key);
                    files.put(id, new KeyFile(id, key, drive));
                    keyFileLocations.add (new KeyFileLocation(
                            KeyFile.getKeyName(key), id, f.getAbsolutePath()));
                  }
                }
              }
            }
          } 
          catch (Exception e) 
          {
            // Ignore this file since it will not contain a key.
          }
        }
      }
    }
  }
}
