package de.das.encrypter.processors;

import de.das.encrypter.tools.BaseThread;
import de.das.encrypter.tools.HexTool;

import java.util.Random;

/**
 * The random factor generator is a tool that can run independently and 
 * concurrently with an application. It permanently generates random floating 
 * point numbers between 0.0 and 1.0 in intervals of 100ms and passes them to 
 * the instance registered as RandomFactorReceiver.
 * 
 * @author Frank Keldenich
 */
public class RandomFactorGenerator extends BaseThread
{
  private Random random;
  
  private final RandomFactorReceiver randomFactorReceiver;
  
  private final byte [] randomInt = new byte [8];
  
  private int randomValue = 0;
  
  private final int intMax = 0x7FFFFFFF;
  
  private static RandomFactorGenerator rfg;
  
  /**
   * Creates an instance of the class RandomFactorGenerator and starts the 
   * underlying thread.
   * 
   * @param rfr instance of a class which implements the interface 
   * RandomFactorReceiver.
   */
  public static void createInstance(RandomFactorReceiver rfr)
  {
    rfg = new RandomFactorGenerator(rfr);
    rfg.start();
  }
  
  /**
   * Creates an instance of the class RandomFactorGenerator.
   * 
   * @param rfr instance of a class which implements the interface 
   * RandomFactorReceiver.
   */
  public RandomFactorGenerator(RandomFactorReceiver rfr)
  {
    randomFactorReceiver = rfr;
    init();
  }

  private void init()
  {
    setSleepTime(100);
    setName("Random Factor Generator");
    random = new Random(System.currentTimeMillis());
  }

  private void nextRandom()
  {
    random.nextBytes (randomInt);
    // Make sure the integer value is positiv.
    randomInt[7] = (byte) (randomInt[7] & 0x7F);
    int additional = HexTool.byteToInt(randomInt);
    randomValue = randomValue + additional;
    // Make sure result is positiv.
    randomValue = randomValue < 0 ? -randomValue : randomValue;
  }

  @Override
  public void doTask()
  {
    nextRandom();
    randomFactorReceiver.setRandomFactor(randomValue / (double)intMax);
  }
}
