package de.das.encrypter.processors;

/**
 * An instance of this class can be used to split a byte array into individual, 
 * smaller blocks. This is necessary if large data packets (larger than 65500 
 * bytes) are to be sent via UDP.<br><br>
 * After creating a new instance, each call to the getNext() method returns a 
 * block of size KeyTransfer.BUFFER_SIZE, except for the last block when the 
 * total size is not an integer multiple of KeyTransfer.BUFFER_SIZE. Then a 
 * smaller byte array is returned with the remainder. When the decomposition 
 * is finished, "getNext()" returns <b>null</b>.
 * 
 * @author Frank Keldenich
 */
public class Splitter 
{
  private byte [] buffer = new byte [KeyTransfer.BUFFER_SIZE];
  
  private final byte [] data;
  
  private final ProgressListener progressListener;
  
  private int currentCount = 0;

  private int blockCount = 0;

  /**
   * Creates a new instance of the Splitter class that splits the passed byte 
   * array into individual blocks of size KeyTransfer.BUFFER_SIZE.
   * 
   * @param data the byte array to be splitted into snaller parts.
   * @param pl a class that implements the ProgressListener interface, or 
   * <b>null</b> if no progress listener is needed.
   */
  public Splitter(byte [] data, ProgressListener pl) 
  {
    this.data = data;
    this.progressListener = pl;
  }

  /**
   * Returns the number of already delivered blocks.
   * @return the number of already delivered blocks.
   */
  public int getBlockCount()
  {
    return blockCount;
  }
  
  /**
   * Returns the next block of data to be split with size 
   * KeyTransfer.BUFFER_SIZE or smaller if it is the last block and the total 
   * size of the data to be split is not an integer multiple of 
   * KeyTransfer.BUFFER_SIZE, or <b>null</b> if all blocks have already been 
   * supplied.
   * 
   * @return the next block of data or <b>null</b> if all blocks have already 
   * been supplied.
   */
  public byte [] getNext()
  {
    if (currentCount == data.length)
    {
      return null;
    }
    if (data.length - currentCount < buffer.length)
    {
      buffer = new byte [data.length - currentCount];
    }
    System.arraycopy(data, currentCount, buffer, 0, buffer.length);
    currentCount = currentCount + buffer.length;
    if (progressListener != null)
    {
      progressListener.setCurrentAmount(currentCount);
    }
    blockCount++;
    return buffer;
  }
  
}
