package de.das.encrypter.processors;

import de.das.encrypter.tools.HexTool;

import de.das.encrypter.comm.DataReceiver;
import de.das.encrypter.comm.UdpSender;

import java.util.Arrays;

/**
 * Class for managing the transmission process between a block sender and the 
 * associated receiver. It implements the DataReceiver interface
 *
 * @author Frank Keldenich
 */
public class TransferControl implements DataReceiver
{
  /**
   * String used for testing whether a receiver is available. Together with this 
   * string the ID of the current key is sent and the recipient replies with 
   * "Ready:OK" if he also has exactly this key, otherwise with "Ready:NOK".
   */
  public static final String PARTNER_ID = "Hello partner:";
  
  /**
   * String that the receiver sends as a response if it was addressed by the 
   * sender with the PARTNER_ID + key ID. It completes the response with "OK" 
   * if it also possesses the key with the sent key ID and with "NOK" if it does 
   * not possess this key. 
   */
  public static final String PARTNER_RESPONSE = "Ready:";
  
  /**
   * String that the receiver sends as a response if errors occur during 
   * reception and evaluation or use of the received data and regular reception 
   * cannot be performed.
   */
  public static final String TRANSFER_BREAK = "Break";
  
  /**
   * This flag indicates that not yet any result has been detected.
   */
  public static final int NO_RESULT = -1;
  
  /**
   * The result of a performed transfer indicates that the transfer was 
   * performed successfully.
   */
  public static final int SUCCESS = 0;
  
  /**
   * The result of a performed transfer indicates that the transfer was 
   * performed with an error.
   */
  public static final int FAILURE = 1;
  
  private UdpSender sender;
  
  private int frameId;
  
  private boolean response;

  private boolean waitForResponse;
    
  private boolean nextFrame;
  
  private boolean breakSender;
  
  private boolean waitingForInfo;
  
  private final ResultReceiver results;
  
  private int transferResult;

  /**
   * Creates a new instance of a transfer control.
   * 
   * @param rr an instance of a class which implements the interface 
   * ResultReceiver.
   */
  public TransferControl(ResultReceiver rr)
  {
    this.results = rr;
    init();
  }

  /**
   * 
   * @param host
   * @param port
   * @throws Exception 
   */
  public void setupSender (String host, int port) throws Exception
  {
    sender = new UdpSender("Transfer Control", host, port);
  }
  
  private void init()
  {
    
  }
  
  public void setHost (String host, int port) throws Exception
  {
    setupSender(host, port);
  }
  
  public boolean isResponse() 
  {
    return response;
  }

  public void setResponse(boolean response) 
  {
    this.response = response;
  }

  public boolean isWaitForResponse() 
  {
    return waitForResponse;
  }

  public void setWaitForResponse(boolean waitForResponse) 
  {
    this.waitForResponse = waitForResponse;
  }

  public boolean isNextFrame() 
  {
    return nextFrame;
  }

  public void setNextFrame(boolean nextFrame) 
  {
    this.nextFrame = nextFrame;
  }

  public int getFrameId() 
  {
    return frameId;
  }

  public void setFrameId(int frameId) 
  {
    this.frameId = frameId;
  }

  public boolean isBreakSender() 
  {
    return breakSender;
  }

  public void setBreakSender(boolean breakSender) 
  {
    this.breakSender = breakSender;
  }

  public UdpSender getSender() 
  {
    return sender;
  }

  public boolean isWaitingForInfo() 
  {
    return waitingForInfo;
  }

  public void setWaitingForInfo(boolean b) 
  {
    this.waitingForInfo = b;
  }

  @Override
  public void setData(String host, byte[] bytes) 
  {
    setData(bytes);
  }
  
  @Override
  public void setData(byte[] data) 
  {
    if (waitingForInfo)
    {
      if (new String (data).startsWith(PARTNER_RESPONSE))
      {
        String info = new String (data).substring(PARTNER_RESPONSE.length());
        breakSender = true;
        results.setResult(ResultReceiver.PARTNER_RESPONSE, info);
      }
      waitingForInfo = false;
    }
    else if (isWaitForResponse())
    {
      if (data.length == KeyTransfer.FAIL.length &&
          Arrays.equals(data, KeyTransfer.FAIL))
      {
        results.setResult(ResultReceiver.TRANSFER_FAILED, null);
      }
      if (data.length == KeyTransfer.VALID.length &&
          Arrays.equals(data, KeyTransfer.VALID))
      {
        results.setResult(ResultReceiver.TRANSFER_VALID, null);
      }
      if (data.length == TRANSFER_BREAK.length() &&
          new String(data).equals(TRANSFER_BREAK))
      {
        results.setResult(ResultReceiver.TRANSFER_BREAK, null);
      }
      else if (data.length == 4 &&
          HexTool.byteToInt(data) == getFrameId())
      {
//System.out.println ("TrCtrl: " + HexTool.byteToInt(data));
        setNextFrame(true);
      }
      setResponse(true);
    }
  }

  /**
   * Sets the result of the performed transfer.
   * @param res the result of the performed transfer.
   */
  public void setTransferResult(int res) 
  {
    transferResult = res;
  }
  
  /**
   * Returns the result of a performed transfer.
   * @return the result of a performed transfer.
   */
  public int getTransferResult()
  {
    return transferResult;
  }
}
