package de.das.encrypter.tools;

import java.io.File;
import java.io.RandomAccessFile;

/**
 * This class provides the functionality of a buffered byte array. Instead of 
 * loading all data into memory, the data is managed in an external file and 
 * only loaded into memory in portions, so that array sizes of 100MB and larger 
 * can also be worked with without causing out-of-memory exceptions.
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class BufferedByteArray 
{
  private static final int DEFAULT_BUFFER_SIZE = 1024 * 1024;
  
  /** 
   * During development
   */
  private static final int MINIMUM_BUFFER_SIZE = 32;
  
//  private static final int MINIMUM_BUFFER_SIZE = 1024;
  
  private static final int MAXIMUM_BUFFER_SIZE = 10 * DEFAULT_BUFFER_SIZE;
  
  private byte [] buffer;
  
  RandomAccessFile dataFile;
  
  private int maxBufferPosition;
  
  private int block;
  
  private long bufferedByteArraySize;
  
  private boolean dataChanges = false;
  
  public BufferedByteArray(File f) throws Exception
  {
    this (f, DEFAULT_BUFFER_SIZE);
  }
  
  public BufferedByteArray(File f, int bufferSize) throws Exception
  {
    this.dataFile = new RandomAccessFile(f, "rw");
    init(bufferSize);
  }

  private void init(int bufferSize) throws Exception
  {
    if (bufferSize < MINIMUM_BUFFER_SIZE || bufferSize > MAXIMUM_BUFFER_SIZE)
    {
      throw new Exception ("Invalid buffer size");
    }
    bufferedByteArraySize = dataFile.length();
    buffer = new byte [bufferSize];
    block = 0;
    
    maxBufferPosition = 0;
    if (dataFile.length() > 0)
    {
      maxBufferPosition = dataFile.read(buffer);
    }
  }

  public byte getByteAt (int position) throws Exception
  {
    if (position >= bufferedByteArraySize)
    {
      throw new Exception ("Index out of range.");
    }
    int neededBlock = position / buffer.length;
    int pointer = position % buffer.length;
    if (neededBlock != block)
    {
      if (dataChanges)
      {
        dataFile.write(buffer, block * buffer.length, maxBufferPosition);
      }
      dataFile.seek(neededBlock * buffer.length);
      maxBufferPosition = dataFile.read(buffer);
      block = neededBlock;
    }
    return buffer [pointer];
  }
  
  public long size()
  {
    return bufferedByteArraySize;
  }
  
  /**
   * Copies the given amount of bytes from this buffered byte array to the given 
   * target byte array starting at specified target position.
   * 
   * @param pos specifies the starting position of this buffered byte array.
   * @param target specifies a byte array to copy the data to. 
   * @param targetPos specifies the start position in the target byte array.
   * @param length specifies the amount of bytes to be copied.
   * 
   * @throws Exception if the index is out of range for one of the byte arrays.
   */
  public void arrayCopy (int pos, byte [] target, int targetPos, int length) throws Exception
  {
    boolean fail = false;
    if (pos + length > bufferedByteArraySize || targetPos + length > target.length)
    {
      fail = true;
    }
    if (fail)
    {
      throw new Exception ("Boundary violation at one of the involved byte arrays.");
    }
    int neededBlock = pos / buffer.length;
    if (neededBlock != block)
    {
      if (dataChanges)
      {
        dataFile.write(buffer, block * buffer.length, maxBufferPosition);
      }
      dataFile.seek(neededBlock * buffer.length);
      maxBufferPosition = dataFile.read(buffer);
      block = neededBlock;
    }
    // Determine whether the array copy can be executed immediately without 
    // buffer changes.
    int end = pos + length;
    while (pos < end)
    {
      int relPos = pos % buffer.length;
      if (end - pos < maxBufferPosition)
      {
        System.arraycopy(buffer, relPos, target, targetPos, end - pos);
        end = pos;
      }
      else
      {
        System.arraycopy(buffer, relPos, target, targetPos, buffer.length - relPos);
        pos = pos + (buffer.length - relPos);
        targetPos = targetPos + (buffer.length - relPos);
        relPos = 0;
        block++;
        maxBufferPosition = dataFile.read(buffer);
      }
    }
  }
}
