package de.das.encrypter.tools;

/**
 * This class provides a converter to build string from byte arrays and vice 
 * versa, where the byte values greater than "0x7F" will be replaced by the
 * values where the MSB of the byte is set to "0" and the MSB itself is placed
 * as a bit into an appended table.
 * 
 * @author Frank Keldenich
 */
public class ByteArrayConverter 
{
  /**
   * An identifier to indicate that a string represents a byte array.
   */
  public static final String IDENTIFIER = "##@@";
  
  /**
   * Converts the given byte array into a string consisting of an identifier, a 
   * four-byte string representing the number of pure binary data plus four-byte 
   * CRC value, the reduced binary data plus CRC value and a table with the 
   * removed MSB bits.
   * 
   * @param data the given binary data.
   * @return the converted string.
   * @throws Exception in case of any error.
   */
  public String byteArrayToString (byte [] data) throws Exception
  {
    String s = IDENTIFIER;
    int crc = (int) CRC.getCRC_32(data);
    byte [] dataPlus = new byte [data.length + 4];
    System.arraycopy(data, 0, dataPlus, 0, data.length);
    HexTool.insertIntLittleEndianAt(crc, dataPlus.length - 4, dataPlus);
    int tableSize = dataPlus.length / 7;
    tableSize = dataPlus.length % 7 == 0 ? tableSize : tableSize + 1;
    byte [] msbTable = new byte [tableSize];

    int bytePointer = 0;
    byte bitMask = 0x40;
    for (int pointer = 0; pointer < dataPlus.length; pointer++)
    {
      if (dataPlus[pointer] < 0)
      {
        dataPlus[pointer] = (byte) (dataPlus[pointer] & 0x7F);
        msbTable[bytePointer] = (byte) (msbTable[bytePointer] | bitMask);
      }
      bitMask = (byte) (bitMask >> 1);
      if (bitMask == 0)
      {
        bitMask = 0x40;
        bytePointer++;
      }
    }
    s = s + extend (dataPlus.length, 4) + new String (dataPlus) + new String (msbTable);
    return s;
  }
  
  private String extend (int i, int len)
  {
    String s = Integer.toString(i);
    while (s.length() < 4)
    {
      s = "0" + s;
    }
    return s;
  }
  
  public byte [] stringToByteArray (String str) throws Exception
  {
    if (!str.startsWith(IDENTIFIER))
    {
      throw new Exception("No binaries.");
    }
    str = str.substring(IDENTIFIER.length());
    int bytePlusLength = Integer.parseInt(str.substring(0, 4));
    str = str.substring(4);
    byte [] stringByte = str.getBytes();
    byte [] dataPlus = new byte [bytePlusLength];
    System.arraycopy(stringByte, 0, dataPlus, 0, dataPlus.length);
    byte [] msbTable = new byte [stringByte.length - bytePlusLength];
    System.arraycopy(stringByte, dataPlus.length, msbTable, 0, msbTable.length);

    int bytePointer = 0;
    byte bitMask = 0x40;
    for (int pointer = 0; pointer < dataPlus.length; pointer++)
    {
      if ((msbTable[bytePointer] & bitMask) != 0)
      {
        dataPlus[pointer] = (byte) (dataPlus[pointer] | 0x80);
      }
      bitMask = (byte) (bitMask >> 1);
      if (bitMask == 0)
      {
        bitMask = 0x40;
        bytePointer++;
      }
    }
    byte [] data = new byte [dataPlus.length - 4];
    System.arraycopy(dataPlus, 0, data, 0, data.length);
    int crc = HexTool.byteToIntLittleEndian(bytePlusLength - 4, dataPlus);
    int crcCalc = (int) CRC.getCRC_32(data);
    if (crc != crcCalc)
    {
      throw new Exception ("Conversion CRC error.");
    }
    return data;
  }
}
