package de.das.encrypter.tools;

import de.das.encrypter.processors.ProgressListener;

import java.awt.image.BufferedImage;

import java.io.File;
import java.io.IOException;

import java.util.Arrays;

import javax.imageio.ImageIO;

/**
 * Byte array image enclosers interweave the bits of the individual bytes of a 
 * byte array with the bytes of the image data of an image in PNG format. The 
 * changes to the image content are so minor that they are not visible to the 
 * naked eye. Enclosed can be the bytes of a simple byte array or the bytes of a 
 * file. The difference is that for the bytes of a file an additional identifier 
 * with the original file name is attached. The bits of the data to be hidden 
 * are hidden in the least significant bits of the color components of the 
 * pixels.

 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class ByteArrayImgEncloser extends ByteArrayEncloser
{
  private static int instanceCounter = 0;
  
  private final int [] RGB_MASK = {0x00000001, 0x00000100, 0x00010000};
  
  private ProgressListener listener = null;
  
  private final int thisInstance;

  private boolean breakIt;

  public ByteArrayImgEncloser() 
  {
    this (null);
  }

  public ByteArrayImgEncloser(ProgressListener pl) 
  {
    this.listener = pl;
    instanceCounter++;
    thisInstance = instanceCounter;
  }
  
  /**
   * Checks if the capacity of the image in the given file is sufficient to hold 
   * the given number of bytes.
   * 
   * @param length the length in bytes of a file.
   * @param f the file representing the image to be used as the container.
   * 
   * @return <b>true</b>, if the container is able to hold the data.
   * @throws IOException if the image cannot be opened.
   */
  public boolean canHold(int length, File f) throws IOException
  {
    BufferedImage img = ImageIO.read(f);
    return canHold(length, img);
  }
  
  /**
   * Checks if the capacity of the given image is sufficient to hold 
   * the given number of bytes.
   * 
   * @param length the length in bytes of a file.
   * @param img the image to be used as the container.
   * 
   * @return <b>true</b>, if the container is able to hold the data.
   */
  public boolean canHold(int length, BufferedImage img)
  {
    return length + 4 + ID.length <= getCapacity(img);
  }
  
  /**
   * Calculates the maximum acquisition capacity of a given image under the 
   * condition that each color component (R, G and B) provides the LSB bit for
   * the bits of the data to be woven in.
   * 
   * @param img the image to be used as the container.
   * @return the maximum acquisition capacity of the given image.
   */
  public int getCapacity(BufferedImage img)
  {
    int fileCapacity =  img.getHeight() * img.getWidth() * 3 / 8;
    return fileCapacity;
  }
  
  /**
   * Weaves the given data into the given image.
   * 
   * @param d a given byte array.
   * @param img a given image.
   * @throws Exception when the specified image is not able to capture the 
   * given data.
   */
  public void hideData (byte[] d, BufferedImage img) throws Exception
  {
    byte [] data = new byte [d.length + 4 + ID.length];
    System.arraycopy(d, 0, data, 4 + ID.length, d.length);
    if (!canHold(d.length, img))
    {
      throw new Exception ("Image not large enough.");
    }
    System.arraycopy(ID, 0, data, 0, ID.length);
    HexTool.insertIntLittleEndianAt(d.length, ID.length, data);
    writeBits (img, data);
  }

  /**
   * Samples the bits of each byte of the given byte array from MSB to LSB and 
   * transfers them respectively to the LSB position of each color component of 
   * each pixel of the image in the order R, G and B, sampling the pixels in 
   * horizontal before vertical direction.
   * 
   * @param img the given image to be used as the container.
   * @param data the byte array to be woven into the image.
   */
  private void writeBits(BufferedImage img, byte[] data)
  {
    breakIt = false;
    int x = 0;
    int y = 0;
    int cp = 0;
    int rgb = img.getRGB(x, y);
    for (int i = 0; i < data.length && !breakIt; i++)
    {
      byte b = data[i];
      int mask = 0x80;
      for (int bitPointer = 0; bitPointer < 8; bitPointer++)
      {
        if ((b & mask) != 0)
        {
          rgb = rgb | RGB_MASK[cp];
        }
        else
        {
          rgb = rgb & ~RGB_MASK[cp];
        }
        mask = mask >> 1;
        
        cp++;
        if (cp == 3)
        {
          img.setRGB(x, y, rgb);
          cp = 0;
          x++;
          if (x == img.getWidth())
          {
            x = 0;
            y++;
          }
          rgb = img.getRGB(x, y);
        }
      }
      if (listener != null)
      {
        if (i % 128 == 0)
        {
          listener.setCurrentAmount(i);
        }
      }
    }
    if (listener != null)
    {
      listener.setCurrentAmount(data.length);
    }
    img.setRGB(x, y, rgb);
  }
  
  /**
   * Uses the LSB bits of the R,G, and B components of the pixels of the given 
   * image to reconstruct the desired number of data bytes of a byte array using 
   * the bits to fill the respective bytes from MSB to LSB as the pixels are 
   * scanned in the horizontal before vertical direction.
   * 
   * @param img a given image.
   * @param count the number of bytes to be reconstructed.
   * 
   * @return a byte array of length "count" with the reconstructed data.
   * 
   * @throws Exception in case of any error.
   */
  public byte[] assemble (BufferedImage img, int count) throws Exception
  {
    return assemble(img, count, false);
  }
  
  /**
   * Uses the LSB bits of the R,G, and B components of the pixels of the given 
   * image to reconstruct the desired number of data bytes of a byte array using 
   * the bits to fill the respective bytes from MSB to LSB as the pixels are 
   * scanned in the horizontal before vertical direction.
   * 
   * @param img a given image.
   * @param count the number of bytes to be reconstructed.
   * @param progress a flag indicating that the current amount should be 
   * communicated to the listener.
   * 
   * @return a byte array of length "count" with the reconstructed data.
   * 
   * @throws Exception in case of any error.
   */
  public byte[] assemble (BufferedImage img, int count, boolean progress) throws Exception
  {
    breakIt = false;
    byte [] data = new byte [count];
    byte b;
    int mask;
    int x = 0;
    int y = 0;
    int cp = 0;
    int rgb = img.getRGB(x, y);
    for (int pointer = 0; pointer < count && !breakIt; pointer++)
    {
      mask = 0x80;
      b = 0;
      for (int bitPointer = 0; bitPointer < 8; bitPointer++)
      {
        if ((rgb & RGB_MASK[cp]) != 0)
        {
          b = (byte) (b | mask);
        }
        else
        {
          b = (byte) (b & ~mask);
        }
        mask = mask >> 1;
        cp ++;
        if (cp == 3)
        {
          cp = 0;
          x++;
          if (x == img.getWidth())
          {
            x = 0;
            y++;
          }
          rgb = img.getRGB(x, y);
        }
      }
      data[pointer] = b;
      if (listener != null && progress)
      {
        if (pointer % 128 == 0)
        {
          listener.setCurrentAmount(pointer);
        }
      }
    }
    if (listener != null && progress)
    {
      listener.setCurrentAmount(count);
    }
    return data;    
  }

  /**
   * Assumes the given file contains an image in PNG format and checks, if there 
   * is any woven data.
   * 
   * @param f a given file.
   * @return <b>true</b> if the file contains hidden data.
   * @throws IOException if any problems with the given file.
   * @throws Exception in any other error cases.
   */
  public boolean containsHiddenData(File f) throws IOException, Exception
  { 
    BufferedImage img = ImageIO.read(f);
    return containsHiddenData(img);
  }

  /**
   * Assumes the given file contains an image in PNG format and checks, if there 
   * is any woven data and these data are from a file.
   * 
   * @param f a given file.
   * @return <b>true</b> if the file contains a hidden file.
   * @throws Exception in case of any error.
   */
  public boolean containsHiddenFile(File f) throws Exception
  {
    boolean contains = containsHiddenData(f);
    if (contains)
    {
      BufferedImage img = ImageIO.read(f);
      byte [] hfHeader = assemble (img, ID.length + 4 + HIDDEN_FILE_ID.length + 2);
      byte [] hfId = new byte [4];
      System.arraycopy(hfHeader, ID.length + 4, hfId, 0, hfId.length);
      if (!Arrays.equals(hfId, HIDDEN_FILE_ID))
      {
        contains = false;
      }
    }
    return contains;
  }

  /**
   * Weaves the given data into the given image.
   * 
   * @param data the data to be woven into the image of the specified file.
   * @param f a file in PNG format representing an image.
   * @throws Exception in case of any error.
   */
  public void hideData(byte [] data, File f) throws Exception
  {
    String format = f.getName();
    format = format.substring(format.lastIndexOf(".") + 1).toUpperCase();
    BufferedImage imgx = ImageIO.read(f);
    // We do not know where the image comes from, so we make sure it will be of
    // TYPE_INT_ARGB.
    BufferedImage img = new BufferedImage (imgx.getWidth(), imgx.getHeight(), BufferedImage.TYPE_INT_ARGB);
    img.getGraphics().drawImage(imgx, 0, 0, null);
    
    hideData(data, img);
    
    if (!breakIt)
    {
      ImageIO.write(img, format, f);
    }
    else
    {
      clear (f);
    }
  }

  /**
   * Regains a possible header from the image bits and checks, whether they 
   * contain the required ID.
   * 
   * @param img the image to be tested for hidden data.
   * @return <b>true</b>, if the image contains hidden data.
   * @throws Exception in case of any error.
   */
  public boolean containsHiddenData(BufferedImage img) throws Exception
  {
    // Check, whether data is hidden within the image file.
    byte [] header = assemble(img, ID.length + 4);
    byte [] id = new byte [ID.length];
    System.arraycopy(header, 0, id, 0, id.length);
    return Arrays.equals(ID, id);
  }
  
  /**
   * Restores the data hidden in the image contained in the given file.
   * 
   * @param f a file containing an image in PNG format with hidden data.
   * @return the recovered data.
   * @throws IOException in case of problems with file and image.
   * @throws Exception in case of other errors.
   */
  public byte [] regainData (File f) throws IOException, Exception
  {
    BufferedImage img = ImageIO.read(f);
    return regainData(img);
  }
  
  /**
   * Restores the data hidden in the image.
   * 
   * @param img the image containing hidden data.
   * @return the recovered data.
   * @throws Exception in case of any error.
   */
  public byte [] regainData (BufferedImage img) throws Exception
  {
    byte [] data = null;
    if (containsHiddenData(img))
    {
      // Check, whether data is hidden within the img file.
      byte [] header = assemble(img, ID.length + 4);
      // Get the length of hidden data.
      int length = HexTool.byteToIntLittleEndian(ID.length, header);
      data = new byte [length];
      byte [] imgData = assemble(img, length + header.length, true);
      System.arraycopy(imgData, header.length, data, 0, data.length);
    }
    return data;
  }
  
  /**
   * Get the amount of bytes hidden in the given file.
   * @param f a given file.
   * @return the number of bytes hidden in the given file.
   * @throws Exception in case of any error.
   */
  public int getHiddenArraySize (File f) throws Exception
  {
    BufferedImage img = ImageIO.read(f);
    int length = 0;
    // Get the header.
    byte [] header = assemble(img, ID.length + 4);
    // Get the length of hidden data.
    length = HexTool.byteToIntLittleEndian(ID.length, header);
    return length;
  }
  
  /**
   * Erases the identifier for hidden data.
   * 
   * @param f a given file containing an image.
   * 
   * @throws IOException in case of problems with the image.
   */
  public void clear (File f)  throws IOException
  {
    byte [] CLEAR = {0x00, 0x00, 0x00, 0x00};
    BufferedImage img = ImageIO.read(f);
    writeBits (img, CLEAR);
    ImageIO.write(img, "PNG", f);
  }

  /**
   * Set a flag that indicates to interrupt a running process immediately.
   */
  public void breakIt()
  {
    breakIt = true;
  }

  /**
   * Extracts the file name of the hidden file from the image in the given file.
   * 
   * It is assumed that it was previously checked whether the image contains a 
   * hidden file. 
   * 
   * @param f a given file.
   * @return the file name of the hidden file.
   * @throws IOException in case of issues with the image.
   * @throws Exception in all other cases.
   */
  public String getHiddenFileName(File f) throws IOException, Exception
  {
    BufferedImage img = ImageIO.read(f);
    byte [] prePart = assemble (img, ID.length + 4 + HIDDEN_FILE_ID.length + 2);
    int length = HexTool.byteToShortIntLittleEndian(ID.length + 4 + HIDDEN_FILE_ID.length, prePart);
    byte [] beginning = assemble (img, ID.length + 4 + HIDDEN_FILE_ID.length + 2 + length);
    byte [] nameBytes = new byte [length];
    System.arraycopy(beginning, ID.length + 4 + HIDDEN_FILE_ID.length + 2, nameBytes, 0, length);
    return new String (nameBytes);
  }
  
  /**
   * Reconstructs the desired number of data bytes of a byte array from the 
   * image contained in the given file. This method should be used only if it is 
   * sure that the hidden data is a file content and not just a byte array.
   * 
   * @param f a file containing a PNG image.
   * @param count the number of hidden bytes to be recovered.
   * @return a byte array with the recovered data.
   * 
   * @throws IOException in case of issues with the image.
   * @throws Exception in all other cases.
   */
  public byte [] getContainedBytes (File f, int count) throws IOException, Exception
  {
    BufferedImage img = ImageIO.read(f);
    byte [] data = new byte [count];
    byte [] prePart = assemble (img, ID.length + 4 + HIDDEN_FILE_ID.length + 2);
    int length = HexTool.byteToShortIntLittleEndian(ID.length + 4 + HIDDEN_FILE_ID.length, prePart);
    byte [] beginning = assemble (img, ID.length + 4 + HIDDEN_FILE_ID.length + 2 + length + count);
    System.arraycopy(beginning, ID.length + 4 + HIDDEN_FILE_ID.length + 2 + length, data, 0, count);
    return data;
  }
}
