package de.das.encrypter.tools;

import de.das.encrypter.processors.ProgressListener;

import java.io.File;
import java.io.RandomAccessFile;

import java.util.Arrays;

/**
 * Byte array wav enclosers interweave the bits of the individual bytes of a 
 * byte array with the bytes of the audio data of an audio file in WAV format. 
 * The changes to the audio content are so minor that they are imperceptible to 
 * the naked ear. Enclosed can be the bytes of a simple byte array or the bytes 
 * of a file. The difference is that for the bytes of a file an additional 
 * identifier with the original file name is attached. The bits of the data to 
 * be hidden are hidden in the least significant bits of the volume values in 
 * both stereo channels.
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class ByteArrayWavEncloser extends ByteArrayEncloser
{
  private final int WAV_FILE_HEADER_LENGTH = 44;
  
  private final byte [] leftSample = new byte [2];
  
  private final byte [] rightSample = new byte [2];
  
  private ProgressListener listener = null;
  
  private boolean breakIt;

  public ByteArrayWavEncloser() 
  {
    this (null);
  }

  public ByteArrayWavEncloser(ProgressListener pl) 
  {
    this.listener = pl;
  }
  
  /**
   * Checks if the capacity of the given audio file is sufficient to hold 
   * the given number of bytes.
   * 
   * @param length the number of bytes to be
   * @param f the file to be used as a container.
   * @return <b>true</b>, if the container is able to hold the data.
   */
  public boolean canHold(int length, File f)
  {
    return length + 4 + ID.length <= getCapacity(f);
  }
  
  /**
   * Checks if the capacity of the given audio file is sufficient to hold 
   * the given file. Contrary to the capture of pure data, for files their name 
   * is still placed at the beginning.
   * 
   * @param f the file to be woven into the container.
   * @param audio the file to be used as a container.
   * @return <b>true</b>, if the audio file is able to hold the file data.
   */
  public boolean canHoldFile(File f, File audio)
  {
    return f.length() + 4 + ID.length + 
           2 + HIDDEN_FILE_ID.length + 
           f.getName().getBytes().length <= getCapacity(audio);
  }
  
  
  /**
   * Calculates the maximum acquisition capacity of a given audio file under the 
   * condition that each volume value of both channels provide their LSB bit for
   * the bits of the data to be woven in.
   * 
   * @param container the wav file to be used as the container.
   * @return the maximum acquisition capacity of the given audio file.
   */
  public int getCapacity(File container)
  {
    int fileCapacity = (int) (container.length() - WAV_FILE_HEADER_LENGTH) / 2 / 8;
    return fileCapacity;
  }
  
  
  /**
   * Weaves the given data into the given audio file.
   * 
   * @param d a given byte array.
   * @param container a given audio file.
   * @throws Exception when the specified audio file is not able to capture the 
   * given data.
   */
  public void hideData (byte [] d, File container) throws Exception
  {
    hideData (d, container, true);
  }
  
  public void hideData (byte [] d, File container, boolean withHeader) throws Exception
  {
    breakIt = false;
    byte [] data;
    if (withHeader)
    {
      data = new byte [d.length + 4 + ID.length];
      System.arraycopy(d, 0, data, 4 + ID.length, d.length);
      if (!canHold(d.length, container))
      {
        throw new Exception ("WAV file not large enough.");
      }
      System.arraycopy(ID, 0, data, 0, ID.length);
      HexTool.insertIntLittleEndianAt(d.length, ID.length, data);
    }
    else
    {
      data = d;
    }
    RandomAccessFile in = new RandomAccessFile(container, "rw");
    int filePointer = WAV_FILE_HEADER_LENGTH;
    in.seek(filePointer);
    for (int i = 0; i < data.length && !breakIt; i++)
    {
      byte b = data[i];
      int mask = 0x80;
      for (int bitPointer = 0; bitPointer < 4; bitPointer++)
      {
        in.read(leftSample);
        in.read(rightSample);
        if ((b & mask) != 0)
        {
          leftSample[0] = (byte) (leftSample[0] | 0x01);
        }
        else
        {
          leftSample[0] = (byte) (leftSample[0] & 0xFE);
        }
        mask = mask >> 1;

        if ((b & mask) != 0)
        {
          rightSample[0] = (byte) (rightSample[0] | 0x01);
        }
        else
        {
          rightSample[0] = (byte) (rightSample[0] & 0xFE);
        }
        mask = mask >> 1;
        
        in.seek(filePointer);
        in.write(leftSample);
        in.write(rightSample);
        
        filePointer = filePointer + 4;
      }
      if (listener != null)
      {
        if (i % 128 == 0)
        {
          listener.setCurrentAmount(i);
        }
      }
    }
    if (listener != null)
    {
        listener.setCurrentAmount(data.length);
    }
    in.close();
    
    if (breakIt)
    {
      clear(container);
    }
  }

  public boolean containsHiddenFile(File container) throws Exception
  {
    boolean contains = containsHiddenData(container);
    if (contains)
    {
      try (RandomAccessFile raf = new RandomAccessFile(container, "r")) 
      {
        byte [] hfHeader = assemble (raf, WAV_FILE_HEADER_LENGTH, ID.length + 4 + HIDDEN_FILE_ID.length + 2);
        byte [] hfId = new byte [4];
        System.arraycopy(hfHeader, ID.length + 4, hfId, 0, hfId.length);
        if (!Arrays.equals(hfId, HIDDEN_FILE_ID))
        {
          contains = false;
        }
      }
    }
    return contains;
  }

  public boolean containsHiddenData(File container) throws Exception
  {
    byte[] header;
    // Check, whether data is hidden within the wav file.
    try (RandomAccessFile raf = new RandomAccessFile(container, "r")) 
    {
      // Check, whether data is hidden within the wav file.
      header = assemble(raf, WAV_FILE_HEADER_LENGTH, ID.length + 4);
    }
    byte [] id = new byte [ID.length];
    System.arraycopy(header, 0, id, 0, id.length);
    return Arrays.equals(ID, id);
  }
  
  /**
   * Checks if the specified file contains hidden data and then starts to 
   * recover the data from the file.
   * 
   * @param container a given file assumed it contains hidden data.
   * @return the regained data or <b>null</b>, if the regaining process has been 
   * interrupted.
   * @throws Exception if any error occurs.
   */
  public byte [] regainData (File container) throws Exception
  {
    byte [] data = null;
    if (containsHiddenData(container))
    {
      // Check, whether data is hidden within the wav file.
      try (RandomAccessFile raf = new RandomAccessFile(container, "r")) 
      {
        // Get the header.
        byte [] header = assemble(raf, WAV_FILE_HEADER_LENGTH, ID.length + 4);
        // Get the length of hidden data.
        int length = HexTool.byteToIntLittleEndian(ID.length, header);
        data = new byte [length];
        // Now compile the hidden data and operate the Progress Listener.
        byte [] rafData = assemble(raf, WAV_FILE_HEADER_LENGTH, length + header.length, true);
        if (rafData != null)
        {
          System.arraycopy(rafData, header.length, data, 0, data.length);
        }
      }
    }
    return data;
  }
  
  /**
   * Get the amount of bytes hidden in the given file.
   * @param f a given file.
   * @return the number of bytes hidden in the given file.
   * @throws Exception in case of any error.
   */
  public int getHiddenArraySize (File f) throws Exception
  {
    int length = 0;
    // Check, whether data is hidden within the wav file.
    try (RandomAccessFile raf = new RandomAccessFile(f, "r")) 
    {
      // Get the header.
      byte [] header = assemble(raf, WAV_FILE_HEADER_LENGTH, ID.length + 4);
      // Get the length of hidden data.
      length = HexTool.byteToIntLittleEndian(ID.length, header);
    }
      return length;
  }
  
  public byte [] assemble (File container, int count) throws Exception
  {
    byte[] data = null;
    // Check, whether data is hidden within the wav file.
    try (RandomAccessFile raf = new RandomAccessFile(container, "r")) 
    {
      // Check, whether data is hidden within the wav file.
      byte [] header = assemble(raf, WAV_FILE_HEADER_LENGTH, ID.length + 4);
      byte [] rafData = assemble(raf, WAV_FILE_HEADER_LENGTH, count + header.length);
      if (rafData != null)
      {
        data = new byte [count];
        System.arraycopy(rafData, header.length, data, 0, data.length);
      }
    }
    return data;
  }
  
  private byte[] assemble (RandomAccessFile container, int pointer, int count) throws Exception
  {
    return assemble(container, pointer, count, false);
  }
  
  private byte[] assemble (RandomAccessFile container, int pointer, int count, boolean progress) throws Exception
  {
    breakIt = false;
    byte [] data = new byte [count];
    byte b;
    int mask;
    container.seek(pointer);
    for (int i = 0; i < count && !breakIt; i++)
    {
      mask = 0x80;
      b = 0;
      for (int j = 0; j < 4; j++)
      {
        container.read(leftSample);
        container.read(rightSample);
        if ((leftSample[0] & 0x01) != 0)
        {
          b = (byte) (b | mask);
        }
        else
        {
          b = (byte) (b & ~mask);
        }
        mask = mask >> 1;
        if ((rightSample[0] & 0x01) != 0)
        {
          b = (byte) (b | mask);
        }
        else
        {
          b = (byte) (b & ~mask);
        }
        mask = mask >> 1;
      }
      data[i] = b;
      if (listener != null && progress)
      {
        if (i % 128 == 0)
        {
          listener.setCurrentAmount(i);
        }
      }
    }
    if (listener != null && progress)
    {
      listener.setCurrentAmount(count);
    }
    if (breakIt)
    {
      data = null;
    }
    return data;
  }
  
  public void clear (File container)  throws Exception
  {
    byte [] CLEAR = {0x00, 0x00, 0x00, 0x00};
    hideData (CLEAR, container, false);
  }

  /**
   * Set a flag that indicates to interrupt a running process immediately.
   */
  public void breakIt()
  {
    breakIt = true;
  }
  
  public String getHiddenFileName(File container) throws Exception
  {
    byte [] prePart = assemble (container, HIDDEN_FILE_ID.length + 2);
    int length = HexTool.byteToShortIntLittleEndian(HIDDEN_FILE_ID.length, prePart);
    byte [] beginning = assemble (container, HIDDEN_FILE_ID.length + 2 + length);
    byte [] nameBytes = new byte [length];
    System.arraycopy(beginning, HIDDEN_FILE_ID.length + 2, nameBytes, 0, length);
    return new String (nameBytes);
  }
  
  public byte [] getContainedBytes (File container, int count) throws Exception
  {
    byte [] data = new byte [count];
    byte [] prePart = assemble (container, HIDDEN_FILE_ID.length + 2);
    int length = HexTool.byteToShortIntLittleEndian(HIDDEN_FILE_ID.length, prePart);
    byte [] beginning = assemble (container, HIDDEN_FILE_ID.length + 2 + length + count);
    System.arraycopy(beginning, HIDDEN_FILE_ID.length + 2 + length, data, 0, count);
    return data;
  }
}
