package de.das.encrypter.tools;

import java.io.PrintStream;

/**
 * Methods for handling conversions of data types from and to byte arrays.
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class HexTool 
{

  /**
   * Takes every 4 bytes of the byte array consecutively, converts it to a float 
   * value and places it to its correct place into the float array.
   * 
   * @param byteArray is a given array with bytes representing float values.
   * 
   * @return a float array containing the the float values converted from the 
   * byte array.
   */
  public static float[] byteArrayToFloatArray(byte[] byteArray) 
  {
    float [] floatArray = new float [byteArray.length / 4];
    for (int i = 0; i < floatArray.length; i++)
    {
      floatArray[i] = getFloatAt (i * 4, byteArray);
    }
    return floatArray;
  }

  /**
   * Converts a byte to integer. Unlike simple casting, which would yield the 
   * range [-128 .. 127], this method yields the range [0 .. 255].
   * 
   * @param b a byte value.
   * @return an integer value in the range [0 .. 255].
   */
  public static int byteToInt(byte b) 
  {
    return b < 0 ? 256 + b : b;
  }

  /**
   * Maps the 4 bytes of each float value to its correct place into the byte 
   * array. The resulting byte array will have the size (4 * fArray.length).
   * 
   * @param fArray is a given array with float values.
   * @return a byte array containing the bytes of the float values from the 
   * float array.
   */
  public static byte[] floatArrayToByteArray(float[] fArray) 
  {
    byte [] byteArray = new byte [fArray.length * 4];
    {
      for (int i = 0; i < fArray.length; i++)
      {
        insertFloatAt(i * 4, fArray[i], byteArray);
      }
    }
    return byteArray;
  }

  /**
   * Assigns the four bytes representing a floating point value to a byte array 
   * with bit 31 of the floating point at bit 7 of byte at position "0". 
   * <br><br>
   * Bit 31 (the bit that is selected by the mask 0x80000000) represents the 
   * sign of the floating-point number.<br><br>
   * Bits 30-23 (the bits that are selected by the mask 0x7f800000) represent 
   * the exponent.<br><br>
   * Bits 22-0 (the bits that are selected by the mask 0x007fffff) represent the 
   * significand (sometimes called the mantissa) of the floating-point number
   * .<br><br>
   *  - If the argument is positive infinity, the result is 0x7f800000.<br><br>
   *  - If the argument is negative infinity, the result is 0xff800000.<br><br>
   *  - If the argument is NaN, the result is 0x7fc00000.
   * 
   * @param f a float value.
   * @return a byte array with the four float bytes.
   */
  public static byte[] floatToByteArray(float f) 
  {
    byte [] byteArray = new byte [4];
    insertFloatAt(0, f, byteArray);
    return byteArray;
  }

  /**
   * Converts the given long value into a byte array of length 8. The msb is at 
   * position 7 and the lsb at position 0.
   * @param v is the given long value.
   * @return a byte array of length 8 containing the bytes of the long value.
   * @deprecated Use "longToByteArrayLittleEndian" instead.
   */
  public static byte[] longToByteArray(long v) 
  {
    byte [] b = new byte [8];
    b[7] = (byte)((v & 0xFF00000000000000L) >> 56);
    b[6] = (byte)((v & 0x00FF000000000000L) >> 48);
    b[5] = (byte)((v & 0x0000FF0000000000L) >> 40);
    b[4] = (byte)((v & 0x000000FF00000000L) >> 32);
    b[3] = (byte)((v & 0x00000000FF000000L) >> 24);
    b[2] = (byte)((v & 0x0000000000FF0000L) >> 16);
    b[1] = (byte)((v & 0x000000000000FF00L) >>  8);
    b[0] = (byte)((v & 0x00000000000000FFL));
    return b;
  }

  /**
   * Converts the given long value into a byte array of length 8. The msb is at 
   * position 7 and the lsb at position 0.
   * @param v is the given long value.
   * @return a byte array of length 8 containing the bytes of the long value.
   */
  public static byte[] longToByteArrayLittleEndian(long v) 
  {
    byte [] b = new byte [8];
    b[7] = (byte)((v & 0xFF00000000000000L) >> 56);
    b[6] = (byte)((v & 0x00FF000000000000L) >> 48);
    b[5] = (byte)((v & 0x0000FF0000000000L) >> 40);
    b[4] = (byte)((v & 0x000000FF00000000L) >> 32);
    b[3] = (byte)((v & 0x00000000FF000000L) >> 24);
    b[2] = (byte)((v & 0x0000000000FF0000L) >> 16);
    b[1] = (byte)((v & 0x000000000000FF00L) >>  8);
    b[0] = (byte)((v & 0x00000000000000FFL));
    return b;
  }

  /**
   * Converts the given long value into a byte array of length 8. The msb is at 
   * position 7 and the lsb at position 0.
   * @param v is the given long value.
   * @return a byte array of length 8 containing the bytes of the long value.
   */
  public static byte[] longToByteArrayBigEndian(long v) 
  {
    byte [] b = new byte [8];
    b[0] = (byte)((v & 0xFF00000000000000L) >> 56);
    b[1] = (byte)((v & 0x00FF000000000000L) >> 48);
    b[2] = (byte)((v & 0x0000FF0000000000L) >> 40);
    b[3] = (byte)((v & 0x000000FF00000000L) >> 32);
    b[4] = (byte)((v & 0x00000000FF000000L) >> 24);
    b[5] = (byte)((v & 0x0000000000FF0000L) >> 16);
    b[6] = (byte)((v & 0x000000000000FF00L) >>  8);
    b[7] = (byte)((v & 0x00000000000000FFL));
    return b;
  }
  
  /**
   * Converts the given int value into a byte array of length 4. The msb is at 
   * position 3 and the lsb at position 0 (little endian).
   * @param v is the given int value.
   * @return a byte array of length 4 containing the bytes of the int value.
   */
  public static byte[] intToByteArray(int v) 
  {
    byte [] b = new byte [4];
    b[3] = (byte)((v & 0xFF000000) >> 24);
    b[2] = (byte)((v & 0x00FF0000) >> 16);
    b[1] = (byte)((v & 0x0000FF00) >>  8);
    b[0] = (byte)((v & 0x000000FF));
    return b;
  }

  /**
   * Converts the specified byte array, assumed to be a little endian, to a long 
   * value.
   * 
   * @param b a byte array representing a long value.
   * 
   * @return a long value from the given byte array.
   */
  public static long byteArrayToLong(byte [] b) 
  {
    long v = 
            (((long)b[7]) << 56) & 0xFF00000000000000L | 
            (((long)b[6]) << 48) & 0x00FF000000000000L |
            (((long)b[5]) << 40) & 0x0000FF0000000000L |
            (((long)b[4]) << 32) & 0x000000FF00000000L |
            (((long)b[3]) << 24) & 0x00000000FF000000L |
            (((long)b[2]) << 16) & 0x0000000000FF0000L |
            (((long)b[1]) <<  8) & 0x000000000000FF00L |
            ((long)b[0])         & 0x00000000000000FFL;
    return v;
  }

  /**
   * Creates a string from the bytes of a byte array. If the byte array contains 
   * 0x00 values, only the values up to the position of the first occurrence of 
   * a 0x00 value are used.
   * 
   * @param b a given byte array.
   * @return a string built from the bytes of the given byte array.
   */
  public static String stringFromByteArray(byte[] b) 
  {
    String str = new String (b);
    if (str.contains("\u0000"))
    {
      str = str.substring(0, str.indexOf("\u0000"));
    }
    return str;
  }

  /**
   * Creates a byte array with the bytes representing the characters of the 
   * specified string. The method is equivalent to the "getBytes()" method of 
   * the String class. UTF-8 character set is used for conversion.
   * 
   * @param str a given string.
   * @return a byte array with bytes representing the characters of the string.
   */
  public static byte[] stringToByteArray(String str) 
  {
    byte [] b = str.getBytes();
    return b;
  }
  
  /**
   * Creates a two character string representing the given byte value.
   * @param b a given byte value.
   * @return the string representation of the hex value [00 .. FF]
   */
  public static String toHex (byte b)
  {
    String s = "";
    int lsb = b & 0x0F;
    int msb = (b & 0xF0) >> 4;
    s = s + (msb > 9 ? (char)((msb - 10) + (byte)('A')) : (char)(msb + (byte)('0')));
    s = s + (lsb > 9 ? (char)((lsb - 10) + (byte)('A')) : (char)(lsb + (byte)('0')));
    return s;
  }
  
  /**
   * Creates an eight character string representing the given integer value.
   * @param i a given integer value.
   * @return the string representation of the hex value [00000000 .. FFFFFFFF]
   */
  public static String toTwoHexQuad (int i)
  {
    String s = toHexQuad((short)((i & 0xFFFF0000) >> 16)) + 
               toHexQuad((short)((i & 0x0000FFFF)));
    return s;
  }
  
  /**
   * Creates a four character string representing the given short value.
   * @param sh a given short value.
   * @return the string representation of the hex value [0000 .. FFFF]
   */
  public static String toHexQuad (short sh)
  {
    String s = "";
    int[] b = new int[4];
    b[0] = sh & 0x0F;
    b[1] = (sh & 0xF0) >> 4;
    b[2] = (sh & 0xF00) >> 8;
    b[3] = (sh & 0xF000) >> 12;
    for (int i = 3; i >= 0; i--)
    {
      s = s + (b[i] > 9 ? 
        (char)((b[i] - 10) + (byte)('A')) : 
        (char)(b[i] + (byte)('0')));
    }
    return s;
  }

  /**
   * Converts a string of length two, assumed to be the representation of a hex 
   * value, to a byte. Allowed characters: [0 .. 9, A .. F, a .. f].
   * 
   * @param str a string representing a hex value [00 .. FF].
   * @return the converted byte value.
   * @throws NumberFormatException if one of the characters is not allowed.
   */
  public static byte parseHex(String str) throws NumberFormatException
  {
    if (str.length () != 2)
    {
      throw new NumberFormatException();
    }
    str = str.toUpperCase ();
    byte b = 0;
    
    char c = str.charAt (0);
    if (!(c >= '0' && c <='9' || c >='A' && c <= 'F'))
    {
      throw new NumberFormatException();
    }
    byte msb = c > '9' ? 
            (byte)(((byte)(c) - (byte)'A') + 10) : 
            (byte)((byte)(c) - (byte)'0');

    c = str.charAt (1);
    if (!(c >= '0' && c <='9' || c >='A' && c <= 'F'))
    {
      throw new NumberFormatException();
    }
    int lsb = c > '9' ? 
            (byte)(((byte)(c) - (byte)'A') + 10) : 
            (byte)((byte)(c) - (byte)'0');
    
    b = (byte)(msb * 16 + lsb);
    return b;
  }
  
  /**
   * Expands the string "s" if its length is smaller than the given length "n" 
   * by adding leading characters "c".
   * 
   * @param s the given string.
   * @param n the requested length.
   * @param c the character with which the string is filled.
   * @return the expanded string. 
   */
  public static String fillLeadingPositions (String s, int n, char c)
  {
    while (s.length () < n)
    {
      s = c + s;
    }
    return s;
  }


  /**
   * Converts a string of length four, assumed to be the representation of a hex 
   * value, to a byte. Allowed characters: [0 .. 9, A .. F, a .. f].
   * 
   * @param str a string representing a hex value [0000 .. FFFF].
   * @return the converted byte value.
   * @throws NumberFormatException if one of the characters is not allowed.
   */
  public static short parseQuadHex (String str) throws NumberFormatException
  {
    if (str.length () > 4)
    {
      throw new NumberFormatException();
    }
    while (str.length() < 4)
    {
      str = "0" + str;
    }
    int msb = parseHex(str.substring (0, 2)); 
    int lsb = parseHex(str.substring (2, 4)); 
    return byteToWord (msb, lsb);
  }
  
  /**
   * Generates a short integer value by composing it from the specified parts 
   * (msb and lsb).
   * 
   * @param msb the msb part of the short int value.
   * @param lsb the lsb part of the short int value.
   * @return a short integer value by composing it from the specified parts.
   */
  public static short byteToWord (int msb, int lsb)
  {
    short result = 0x0000;
    lsb = lsb < 0 ? 256 + lsb : lsb;
    msb = msb < 0 ? 256 + msb : msb;
    return (short)((result | msb) << 0x08 | lsb);
  }
  
  /**
   * Generates an integer value by composing it from the specified parts 
   * (msb and lsb).
   * 
   * @param msb the msb part of the int value.
   * @param lsb the lsb part of the int value.
   * @return an integer value by composing it from the specified parts.
   */
  public static int byteToInt (int msb, int lsb)
  {
    lsb = lsb < 0 ? 256 + lsb : lsb;
    msb = msb < 0 ? 256 + msb : msb;
    return msb << 0x08 | lsb;
  }

  
  /**
   * Generates an integer value by composing it from the specified parts 
   * (msb, csb and lsb).
   * 
   * @param msb the msb part of the int value.
   * @param csb the csb part of the int value.
   * @param lsb the lsb part of the int value.
   * @return an integer value by composing it from the specified parts.
   */
  public static int byteToInt (int msb, int csb, int lsb)
  {
    lsb = lsb < 0 ? 256 + lsb : lsb;
    csb = csb < 0 ? 256 + csb : csb;
    msb = msb < 0 ? 256 + msb : msb;
    return msb << 16 | csb << 8 | lsb;
  }

  /**
   * Converts the specified byte array, assumed to be a little endian, to a long 
   * value.
   * 
   * @param bytes a byte array representing a integer value.
   * 
   * @return an integer value from the given byte array.
   */
  public static int byteToInt (byte [] bytes)
  {
    int [] b = new int [4];
    b[3] = bytes [3];
    b[2] = bytes [2];
    b[1] = bytes [1];
    b[0] = bytes [0];
    b[3] = (b[3] << 24) & 0xFF000000;
    b[2] = (b[2] << 16) & 0x00FF0000;
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return b[3] | b[2] | b[1] | b[0];
  }

  /**
   * Converts the specified byte array, assumed to be a little endian, to a long 
   * int value.
   * 
   * @param bytes a byte array representing a long int value.
   * 
   * @return a long int value from the given byte array.
   */
  public static long byteToLongLittleEndian (byte [] bytes)
  {
    long [] b = new long [8];
    b[7] = bytes [7];
    b[6] = bytes [6];
    b[5] = bytes [5];
    b[4] = bytes [4];
    b[3] = bytes [3];
    b[2] = bytes [2];
    b[1] = bytes [1];
    b[0] = bytes [0];
    b[7] = (b[7] << 56) & 0xFF00000000000000L;
    b[6] = (b[6] << 48) & 0x00FF000000000000L;
    b[5] = (b[5] << 40) & 0x0000FF0000000000L;
    b[4] = (b[4] << 32) & 0x000000FF00000000L;
    b[3] = (b[3] << 24) & 0x00000000FF000000L;
    b[2] = (b[2] << 16) & 0x0000000000FF0000L;
    b[1] = (b[1] <<  8) & 0x000000000000FF00L;
    b[0] =  b[0]        & 0x00000000000000FFL;
    return b[7] | b[6] | b[5] | b[4] | b[3] | b[2] | b[1] | b[0];
  }

  /**
   * Converts an eight-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a little-endian value, to a long-int 
   * value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the eight byte long 
   * part starts.
   * 
   * @return a long int value from the given byte array at the specified 
   * positon.
   */
  public static long byteToLongLittleEndian (byte [] bytes, int pos)
  {
    long [] b = new long [8];
    b[7] = bytes [7 + pos];
    b[6] = bytes [6 + pos];
    b[5] = bytes [5 + pos];
    b[4] = bytes [4 + pos];
    b[3] = bytes [3 + pos];
    b[2] = bytes [2 + pos];
    b[1] = bytes [1 + pos];
    b[0] = bytes [0 + pos];
    b[7] = (b[7] << 56) & 0xFF00000000000000L;
    b[6] = (b[6] << 48) & 0x00FF000000000000L;
    b[5] = (b[5] << 40) & 0x0000FF0000000000L;
    b[4] = (b[4] << 32) & 0x000000FF00000000L;
    b[3] = (b[3] << 24) & 0x00000000FF000000L;
    b[2] = (b[2] << 16) & 0x0000000000FF0000L;
    b[1] = (b[1] <<  8) & 0x000000000000FF00L;
    b[0] =  b[0]        & 0x00000000000000FFL;
    return b[7] | b[6] | b[5] | b[4] | b[3] | b[2] | b[1] | b[0];
  }

  /**
   * Converts an four-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a little-endian value, to an integer 
   * value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the four byte long 
   * part starts.
   * 
   * @return an integer value from the given byte array at the specified 
   * positon.
   */
  public static int byteToIntLittleEndian (int pos, byte [] bytes)
  {
    int [] b = new int [4];
    b[0] = bytes [0 + pos];
    b[1] = bytes [1 + pos];
    b[2] = bytes [2 + pos];
    b[3] = bytes [3 + pos];
    b[3] = (b[3] << 24) & 0xFF000000;
    b[2] = (b[2] << 16) & 0x00FF0000;
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return b[3] | b[2] | b[1] | b[0];
  }

  /**
   * Converts an eight-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a big-endian value, to a long-int 
   * value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the eight byte long 
   * part starts.
   * 
   * @return a long int value from the given byte array at the specified 
   * positon.
   */
  public static long byteToLongBigEndian (byte [] bytes, int pos)
  {
    long [] b = new long [8];
    b[7] = bytes [0 + pos];
    b[6] = bytes [1 + pos];
    b[5] = bytes [2 + pos];
    b[4] = bytes [3 + pos];
    b[3] = bytes [4 + pos];
    b[2] = bytes [5 + pos];
    b[1] = bytes [6 + pos];
    b[0] = bytes [7 + pos];
    b[7] = (b[7] << 56) & 0xFF00000000000000L;
    b[6] = (b[6] << 48) & 0x00FF000000000000L;
    b[5] = (b[5] << 40) & 0x0000FF0000000000L;
    b[4] = (b[4] << 32) & 0x000000FF00000000L;
    b[3] = (b[3] << 24) & 0x00000000FF000000L;
    b[2] = (b[2] << 16) & 0x0000000000FF0000L;
    b[1] = (b[1] <<  8) & 0x000000000000FF00L;
    b[0] =  b[0]        & 0x00000000000000FFL;
    return b[7] | b[6] | b[5] | b[4] | b[3] | b[2] | b[1] | b[0];
  }

  /**
   * Converts an four-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a little-endian value, to an 
   * unsigned integer value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the four byte long 
   * part starts.
   * 
   * @return an unsigned integer value from the given byte array at the 
   * specified positon.
   */
  public static long byteToUintLittleEndian (int pos, byte [] bytes)
  {
    long [] b = new long [4];
    b[0] = bytes [0 + pos];
    b[1] = bytes [1 + pos];
    b[2] = bytes [2 + pos];
    b[3] = bytes [3 + pos];
    b[0] = b[0] < 0 ? 256 + b[0] : b[0];
    b[1] = b[1] < 0 ? 256 + b[1] : b[1];
    b[2] = b[2] < 0 ? 256 + b[2] : b[2];
    b[3] = b[3] < 0 ? 256 + b[3] : b[3];
    b[3] = (b[3] << 24) & 0xFF000000;
    b[2] = (b[2] << 16) & 0x00FF0000;
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return b[3] | b[2] | b[1] | b[0];
  }

  /**
   * Converts an four-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a big-endian value, to an integer 
   * value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the four byte long 
   * part starts.
   * 
   * @return an integer value from the given byte array at the specified 
   * positon.
   */
  public static int byteToIntBigEndian (int pos, byte [] bytes)
  {
    int [] b = new int [4];
    b[3] = bytes [0 + pos];
    b[2] = bytes [1 + pos];
    b[1] = bytes [2 + pos];
    b[0] = bytes [3 + pos];
    b[3] = (b[3] << 24) & 0xFF000000;
    b[2] = (b[2] << 16) & 0x00FF0000;
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return b[3] | b[2] | b[1] | b[0];
  }

  /**
   * Converts an four-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a big-endian value, to an 
   * unsigned integer value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the four byte long 
   * part starts.
   * 
   * @return an unsigned integer value from the given byte array at the 
   * specified positon.
   */
  public static long byteToUintBigEndian (int pos, byte [] bytes)
  {
    long [] b = new long [4];
    b[3] = bytes [0 + pos];
    b[2] = bytes [1 + pos];
    b[1] = bytes [2 + pos];
    b[0] = bytes [3 + pos];
    b[0] = b[0] < 0 ? 256 + b[0] : b[0];
    b[1] = b[1] < 0 ? 256 + b[1] : b[1];
    b[2] = b[2] < 0 ? 256 + b[2] : b[2];
    b[3] = b[3] < 0 ? 256 + b[3] : b[3];
    b[3] = (b[3] << 24) & 0xFF000000;
    b[2] = (b[2] << 16) & 0x00FF0000;
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return b[3] | b[2] | b[1] | b[0];
  }

  /**
   * Converts an two-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a little-endian value, to an integer 
   * value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the two byte long 
   * part starts.
   * 
   * @return an integer value from the given byte array at the specified 
   * positon.
   */
  public static int byteToWordLittleEndian(int pos, byte[] bytes)
  {
    int [] b = new int [2];
    b[0] = bytes [0 + pos];
    b[1] = bytes [1 + pos];
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return b[1] | b[0];
  }

  /**
   * Converts an two-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a big-endian value, to an integer 
   * value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the two byte long 
   * part starts.
   * 
   * @return an integer value from the given byte array at the specified 
   * positon.
   */
  public static int byteToWordBigEndian(int pos, byte[] bytes)
  {
    int [] b = new int [2];
    b[1] = bytes [0 + pos];
    b[0] = bytes [1 + pos];
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return b[1] | b[0];
  }

  /**
   * Converts an two-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a big-endian value, to a short 
   * integer value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the two byte long 
   * part starts.
   * 
   * @return a short integer value from the given byte array at the specified 
   * positon.
   */
  public static short byteToShortIntBigEndian(int pos, byte[] bytes) 
  {
    int [] b = new int [2];
    b[1] = bytes [0 + pos];
    b[0] = bytes [1 + pos];
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return (short) (b[1] | b[0]);
  }

  /**
   * Converts an two-byte long part from the specified byte array starting at 
   * the specified position, assuming it is a little-endian value, to a short 
   * int value.
   * 
   * @param bytes a given byte array.
   * @param pos the position within the byte array where the two byte long 
   * part starts.
   * 
   * @return a short integer value from the given byte array at the specified 
   * positon.
   */
  public static short byteToShortIntLittleEndian(int pos, byte[] bytes) 
  {
    int [] b = new int [2];
    b[1] = bytes [1 + pos];
    b[0] = bytes [0 + pos];
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    return (short) (b[1] | b[0]);
  }

  /**
   * Creates a string using the specified part of the given byte array. The part
   * is specified by a starting index and the number of byte to be used.
   * 
   * @param data a given byte array.
   * @param start a starting index.
   * @param length the number of byte to be used.
   * @return a string built from a part of the byte array.
   */
  public static String byteToString (byte [] data, int start, int length)
  {
    String result = "";
    for (int i = start; i < start + length; i++ )
    {
      result = result + (char)(data[i]);
    }
    return result;
  }
  
  /**
   * Generates a string consisting of the string representations of the 
   * hexadecimal values of the given ByteArray separated by spaces.
   * 
   * @param data a given byte array.
   * @return a string with the hex values of the given byte array.
   */
  public static String byteArrayToString (byte [] data)
  {
    return byteArrayToString(data, 0);
  }
  
  /**
   * Generates a string consisting of the string representations of the 
   * hexadecimal values of the given ByteArray separated by spaces. If the block 
   * size "n" is greater than "0", hyphens are inserted between blocks with "n" 
   * hex numbers.
   * 
   * @param data a given byte array.
   * @param blockSize number of hex values per block.
   * @return a string with the hex values of the given byte array.
   */
  public static String byteArrayToString (byte [] data, int blockSize)
  {
    String str = "";
    int count = 0;
    for (byte b: data)
    {
      str = str + toHex (b) + " ";
      if (blockSize > 0)
      {
        if (count % blockSize == blockSize - 1)
        {
          str = str + "- ";
        }
      }
      count++;
    }
    return str;
  }

  /**
   * Assigns the four bytes of an integer value to the bytes of the given byte 
   * array starting at the specified position with byte order big endian.
   * 
   * @param v a given integer value.
   * @param pos the starting position in the byte array.
   * @param data a given byte array.
   */
  public static void insertIntAt(int v, int pos, byte[] data) 
  {
    data [pos + 3] = (byte) (v  & 0x000000FF);
    data [pos + 2] = (byte)((v  & 0x0000FF00) >> 8);
    data [pos + 1] = (byte)((v  & 0x00FF0000) >> 16);
    data [pos + 0] = (byte)((v  & 0xFF000000) >> 24);
  }

  /**
   * Assigns the four bytes of a long value to the bytes of the given byte 
   * array starting at the specified position with byte order big endian.
   * 
   * @param v a given long value.
   * @param pos the starting position in the byte array.
   * @param data a given byte array.
   * @deprecated Use insertIntBigEndianAt instead.
   */
  public static void insertIntAt(long v, int pos, byte[] data) 
  {
    data [pos + 3] = (byte) (v  & 0x000000FF);
    data [pos + 2] = (byte)((v  & 0x0000FF00) >> 8);
    data [pos + 1] = (byte)((v  & 0x00FF0000) >> 16);
    data [pos + 0] = (byte)((v  & 0xFF000000) >> 24);
  }
  
  /**
   * Assigns the four lsb bytes of a long value to the bytes of the given byte 
   * array starting at the specified position with byte order big endian.
   * 
   * @param v a given long value.
   * @param pos the starting position in the byte array.
   * @param data a given byte array.
   */
  public static void insertIntBigEndianAt(long v, int pos, byte[] data) 
  {
    data [pos + 3] = (byte) (v  & 0x000000FF);
    data [pos + 2] = (byte)((v  & 0x0000FF00) >> 8);
    data [pos + 1] = (byte)((v  & 0x00FF0000) >> 16);
    data [pos + 0] = (byte)((v  & 0xFF000000) >> 24);
  }

  /**
   * Assigns the four lsb bytes of a long value to the bytes of the given byte 
   * array starting at the specified position with byte order little endian.
   * 
   * @param v a given long value.
   * @param pos the starting position in the byte array.
   * @param data a given byte array.
   */
  public static void insertIntLittleEndianAt(long v, int pos, byte[] data) 
  {
    data [pos + 0] = (byte) (v  & 0x000000FF);
    data [pos + 1] = (byte)((v  & 0x0000FF00) >> 8);
    data [pos + 2] = (byte)((v  & 0x00FF0000) >> 16);
    data [pos + 3] = (byte)((v  & 0xFF000000) >> 24);
  }

  /**
   * Assigns the eight bytes of a long value to the bytes of the given byte 
   * array starting at the specified position with byte order little endian.
   * 
   * @param v a given long value.
   * @param pos the starting position in the byte array.
   * @param data a given byte array.
   */
  public static void insertLongLittleEndianAt(long v, int pos, byte[] data) 
  {
    data [pos + 0] = (byte) (v  & 0x00000000000000FFL);
    data [pos + 1] = (byte)((v  & 0x000000000000FF00L) >> 8);
    data [pos + 2] = (byte)((v  & 0x0000000000FF0000L) >> 16);
    data [pos + 3] = (byte)((v  & 0x00000000FF000000L) >> 24);
    data [pos + 4] = (byte)((v  & 0x000000FF00000000L) >> 32);
    data [pos + 5] = (byte)((v  & 0x0000FF0000000000L) >> 40);
    data [pos + 6] = (byte)((v  & 0x00FF000000000000L) >> 48);
    data [pos + 7] = (byte)((v  & 0xFF00000000000000L) >> 56);
  }

  /**
   * Assigns the two lsb bytes of an int value to the bytes of the given byte 
   * array starting at the specified position with byte order big endian.
   * 
   * @param v a given integer value.
   * @param pos the starting position in the byte array.
   * @param data a given byte array.
   * @deprecated Use insertShortIntBigEndianAt instead.
   */
  public static void insertShortIntAt(int v, int pos, byte [] data) 
  {
    data [pos + 1] = (byte)(v  & 0x000000FF);
    data [pos + 0] = (byte)((v  & 0x0000FF00) >> 8);
  }

  /**
   * Assigns the two lsb bytes of an int value to the bytes of the given byte 
   * array starting at the specified position with byte order big endian.
   * 
   * @param v a given integer value.
   * @param pos the starting position in the byte array.
   * @param data a given byte array.
   */
  public static void insertShortIntBigEndianAt(int v, int pos, byte [] data) 
  {
    data [pos + 1] = (byte)(v  & 0x000000FF);
    data [pos + 0] = (byte)((v  & 0x0000FF00) >> 8);
  }


  /**
   * Assigns the two lsb bytes of an int value to the bytes of the given byte 
   * array starting at the specified position with byte order little endian.
   * 
   * @param v a given integer value.
   * @param pos the starting position in the byte array.
   * @param data a given byte array.
   */
  public static void insertShortIntLittleEndianAt(int v, int pos, byte [] data) 
  {
    data [pos + 0] = (byte)(v  & 0x000000FF);
    data [pos + 1] = (byte)((v  & 0x0000FF00) >> 8);
  }

  /**
   * Converts a the float four bytes representation into an integer and places
   * the four bytes of the integer into the byte array from the given position 
   * on. 
   * @param pos is the position in the byte array where the float value has to 
   * be placed at.
   * @param fValue is the float value to be placed into the byte array.
   * @param data is the byte array the float value shall be written to.
   */
  public static void insertFloatAt(int pos, float fValue, byte [] data)
  {
    int value = Float.floatToIntBits(fValue);
    data [pos + 3] = (byte)(value  & 0x000000FF);
    data [pos + 2] = (byte)((value  & 0x0000FF00) >> 8);
    data [pos + 1] = (byte)((value  & 0x00FF0000) >> 16);
    data [pos + 0] = (byte)((value  & 0xFF000000) >> 24);
  }

  /**
   * Converts the four bytes at the given starting position to a floating point
   * value assuming the byte order as little endian.
   * 
   * @param pos starting position in the byte array.
   * @param data a given byte array.
   * @return an integer value.
   */
  public static float getFloatAt (int pos, byte [] data)
  {
    return Float.intBitsToFloat(getIntAt(pos, data));
  }

  /**
   * Converts the four bytes at the given starting position to an integer value
   * assuming as little endian.
   * 
   * @param pos starting position in the byte array.
   * @param data a given byte array.
   * @return an integer value.
   */
  public static int getIntAt (int pos, byte [] data)
  {
    int[] b = new int[4];
    b[3] = data [pos + 0];
    b[2] = data [pos + 1];
    b[1] = data [pos + 2];
    b[0] = data [pos + 3];
    b[3] = (b[3] << 24) & 0xFF000000;
    b[2] = (b[2] << 16) & 0x00FF0000;
    b[1] = (b[1] <<  8) & 0x0000FF00;
    b[0] =  b[0]        & 0x000000FF;
    
    int value = b[3] | b[2] | b[1] | b[0];

    return value;
  }

  /**
   * Converts the eight bytes at the given starting position to a long value
   * assuming as little endian.
   * 
   * @param pos starting position in the byte array.
   * @param data a given byte array.
   * @return a long value.
   */
  public static long getLongAt (int pos, byte [] data)
  {
    long[] b = new long[8];
    b[7] = data [pos + 0];
    b[6] = data [pos + 1];
    b[5] = data [pos + 2];
    b[4] = data [pos + 3];
    b[3] = data [pos + 4];
    b[2] = data [pos + 5];
    b[1] = data [pos + 6];
    b[0] = data [pos + 7];
    b[7] = (b[7] << 56) & 0xFF00000000000000L;
    b[6] = (b[6] << 48) & 0x00FF000000000000L;
    b[5] = (b[5] << 40) & 0x0000FF0000000000L;
    b[4] = (b[4] << 32) & 0x000000FF00000000L;
    b[3] = (b[3] << 24) & 0x00000000FF000000L;
    b[2] = (b[2] << 16) & 0x0000000000FF0000L;
    b[1] = (b[1] <<  8) & 0x000000000000FF00L;
    b[0] =  b[0]        & 0x00000000000000FFL;
    
    long value = b[7] | b[6] | b[5] | b[4] | b[3] | b[2] | b[1] | b[0];

    return value;
  }

  /**
   * Converts the byte values of the byte array into hex strings and arranges 
   * them one after another separated by spaces.
   * @param data a given byte array.
   * @return a string with hex string separated by spaces.
   */
  public static String getHexString(byte[] data)
  {
    String s = "";
    for (byte b: data)
    {
      s = s + toHex(b) + " ";
    }
    return s;
  }

  /**
   * Converts the specified byte array to a hex string and outputs it over the 
   * specified print stream.
   * 
   * @param out a given print stream.
   * @param data a given byte array.
   */
  public static void printBuffer(PrintStream out, byte[] data)
  {
    out.print (getHexString(data));
  }

  /**
   * Converts the specified byte array to a hex string and outputs it over the 
   * System.out print stream.
   * 
   * @param data a given byte array.
   */
  public static void printBuffer (byte [] data)
  {
    printBuffer(System.out, data);
  }

  /**
   * Converts the specified byte array to a hex string and outputs it over the 
   * System.out print stream followed by a System.out.println().
   * 
   * @param data a given byte array.
   */
  public static void printlnBuffer (byte [] data)
  {
    printBuffer (data);
    System.out.println();
  }
  
  
  /**
   * Generates a string representing a formatted hex list with sixteen bytes per 
   * line and a preceding eight-character hexadecimal address. The start value
   * for the addresses is assumed to be zero. The string inserted between the 
   * eighth and ninth hexadecimal value in each line is assumed to two spaces.
   * 
   * @param data a given byte array.
   * @return a string representing a formatted hex list.
   */
  public static String createFormattedHexList (byte [] data)
  {
    return createFormattedHexList(0, data, "  ");
  }
  
  /**
   * Generates a string representing a formatted hex list with sixteen bytes per 
   * line and a preceding eight-character hexadecimal address. The start value
   * for the addresses is assumed to be zero.
   * 
   * @param data a given byte array.
   * @param centerDiv a string inserted between the eighth and ninth hexadecimal 
   * value in each line.
   * @return a string representing a formatted hex list.
   */
  public static String createFormattedHexList (byte [] data, String centerDiv)
  {
    return createFormattedHexList(0, data, centerDiv);
  }
  
  /**
   * Generates a string representing a formatted hex list with sixteen bytes per 
   * line and a preceding eight-character hexadecimal address.
   * 
   * @param adr a given start value for the preceding address.
   * @param data a given byte array.
   * @param centerDiv a string inserted between the eighth and ninth hexadecimal 
   * value in each line.
   * @return a string representing a formatted hex list.
   */
  public static String createFormattedHexList (int adr, byte [] data, String centerDiv)
  {
    String line;
    byte [] page = new byte [128];
    int pages = data.length / 128;
    if (pages * 128 < data.length)
    {
      pages++;
    }
    
    byte [] byteOctalPackage = new byte [8];
    short adrLow = (short) (adr & 0xFFFF);
    short adrHigh = (short)((adr & 0xFFFF0000) >> 16);
    line = "";
    boolean breakNow = false;
    
    breakHere:
    for (int j = 0; j < pages; j++)
    {
      line = line + "\n";
      int byteCount = 128;
      if (data.length - j * 128 < 128)
      {
        byteCount = data.length - j * 128;
      } 
      System.arraycopy(data, j * 128, page, 0, byteCount);
      for (int octalCounter = 0; octalCounter < 16; octalCounter++)
      {
        int n = j * 128 + (octalCounter + 1) * 8;
        if (n > data.length)
        {
          byteOctalPackage = new byte [8 - (n - data.length)];
          System.arraycopy(
                  page, octalCounter * 8, 
                  byteOctalPackage, 0, 
                  byteOctalPackage.length);
          breakNow = true;
        }
        else
        {
          System.arraycopy(page, octalCounter * 8, byteOctalPackage, 0, 8);
        }
        if (octalCounter - octalCounter / 2 * 2 == 0)
        {
          line = line + "\n" + 
                  HexTool.toHexQuad((short)(adrHigh)) + 
                  HexTool.toHexQuad((short)(adrLow)) + ": ";
          adrHigh = adrLow + 16 > 0xFFFF ? (short)(adrHigh + 1) : adrHigh;
          adrLow = (short) (adrLow + 16);
        }
        else
        {
          line = line + centerDiv;
        }
        line = line + HexTool.byteArrayToString(byteOctalPackage);
        if (breakNow)
        {
          break breakHere;
        }
      }
    }
    return line;
  }
  
  /**
   * Reverses the order of the bytes in a byte array.
   * 
   * @param data a given byte array.
   * @return the byte array where the order of the bytes was reversed.
   */
  public static byte [] swapBytes(byte [] data)
  {
    byte tmp;
    for (int i = 0; i < data.length / 2; i++)
    {
      tmp = data [i];
      data[i] = data[data.length - i - 1];
      data[data.length - i - 1] = tmp;
    }
    return data;
  }

}
