package de.das.encrypter.tools;

import com.google.zxing.BinaryBitmap;
import com.google.zxing.ChecksumException;
import com.google.zxing.FormatException;
import com.google.zxing.NotFoundException;
import com.google.zxing.RGBLuminanceSource;
import com.google.zxing.Result;

import com.google.zxing.common.HybridBinarizer;

import com.google.zxing.qrcode.QRCodeReader;

import de.das.encrypter.model.Key;
import de.das.encrypter.model.KeyFile;

import java.awt.image.BufferedImage;

import java.io.File;

import java.util.Arrays;

import javax.imageio.ImageIO;

/**
 * This class provides methods for hiding and recovering keys in and from PNG 
 * images, WAV files and QR images.
 * 
 * @author Frank Keldenich
 */
public class HiddenDataTool 
{
  private final ByteArrayWavEncloser bawe = new ByteArrayWavEncloser();
  
  private final ByteArrayImgEncloser baie = new ByteArrayImgEncloser();
  
  /**
   * Recovers a key contained in a QR code image.
   * 
   * @param f a file containing the image of a QR code.
   * @return the key that was hidden in the QR code or <b>null</b> if no key was 
   * found.
   */
  public byte [] keyFromQr(File f)
  {
    byte [] data = byteArrayFromQr(f);
    if (data != null && !KeyFile.isValidKey(data))
    {
      data = null;
    }
    return data;
  }
  
  /**
   * Checks whether a key is contained the QR-Code contained in the 
   * specified file.
   * @param f a given file.
   * @return <b>true</b>, if a key is contained in the QR-Code.
   * @throws Exception in case of an error.
   */
  public boolean keyInQr(File f) throws Exception
  {
    boolean contains = false;
    byte [] data = byteArrayFromQr(f);
    if (data != null)
    {
      byte [] keyfileId = new byte [8];
      System.arraycopy(data, 0, keyfileId, 0, keyfileId.length);
      contains = Arrays.equals(keyfileId, Key.KEY_SIGNATURE_PREFIX.getBytes());
    }
    return contains;
  }

  /**
   * Checks whether a key is woven into the audio stream contained in the 
   * specified file.
   * @param f a given file.
   * @return <b>true</b>, if a key is woven into the audio stream.
   * @throws Exception in case of an error.
   */
  public boolean keyInWav(File f) throws Exception
  {
    boolean contains = false;
    // Check, if it is a WAV file.
    if (bawe.containsHiddenData(f))
    {
      byte [] keyfileId = bawe.assemble(f, 8);
      contains = Arrays.equals(keyfileId, Key.KEY_SIGNATURE_PREFIX.getBytes());
    }
    return contains;
  }

  /**
   * Checks whether a key is woven into the image contained in the specified
   * file.
   * @param f a given file.
   * @return <b>true</b>, if a key is woven into the image.
   * @throws Exception in case of an error.
   */
  public boolean keyInPng(File f) throws Exception
  {
    boolean contains = false;
    // Check, if it is a PNG file.
    if (baie.containsHiddenData(f))
    {
      BufferedImage img = ImageIO.read(f);
      byte [] data = baie.assemble(img, 32);
      byte [] keyfileId = new byte [8];
      System.arraycopy(data, 4 + ByteArrayImgEncloser.ID.length, keyfileId, 0, keyfileId.length);
      contains = Arrays.equals(keyfileId, Key.KEY_SIGNATURE_PREFIX.getBytes());
    }
    return contains;
  }

  public byte [] byteArrayFromImg(File f)
  {
    byte [] data = null;
    try
    {
      BufferedImage img = ImageIO.read(f);
      if (img != null && baie.containsHiddenData(img))
      {
        data = baie.regainData(img);
      }
    }
    catch (Exception x)
    {

    }
    return data;
  }

  public byte [] byteArrayFromWav(File f)
  {
    byte [] data = null;
    try
    {
      if (bawe.containsHiddenData(f))
      {
        data = bawe.regainData(f);
      }
    }
    catch (Exception x)
    {

    }
    return data;
  }

  public byte [] byteArrayFromQr(File f)
  {
    byte [] data = null;
    try
    {
      BufferedImage img = ImageIO.read(f);
      if (img != null)
      {
        String content = readQRCode(img);
        if (content != null)
        {
          ByteArrayConverter bac = new ByteArrayConverter ();
          data = bac.stringToByteArray(content);
        }
      }
    }
    catch (Exception x)
    {

    }
    return data;
  }
    
  private String readQRCode(BufferedImage image) throws Exception
  {
    BinaryBitmap bitmap;
    Result result;
    int[] pixels = image.getRGB(0, 0, image.getWidth(), image.getHeight(), null, 0, image.getWidth());
    RGBLuminanceSource source = new RGBLuminanceSource(image.getWidth(), image.getHeight(), pixels);
    bitmap = new BinaryBitmap(new HybridBinarizer(source));

    QRCodeReader reader = new QRCodeReader();	
    try 
    {
      result = reader.decode(bitmap);
      return result.getText();
    } 
    catch (NotFoundException e) 
    {
    } 
    catch (ChecksumException e) 
    {
    } 
    catch (FormatException e) 
    {
    }

    return null;
   }

  public String getKeyNameFromWav(File f) 
  {
    String name = "";
    try
    {
      // Check, if it is a WAV file.
      if (bawe.containsHiddenData(f))
      {
        byte [] keyfileId = bawe.assemble(f, KeyFile.HEADER_LENGTH_PLUS_NAME_AMOUNT);
        byte [] amountBytes = new byte [2]; 
        System.arraycopy(keyfileId, KeyFile.HEADER_LENGTH, amountBytes, 0, amountBytes.length);
        byte nameLength = HexTool.parseHex(new String(amountBytes));
        keyfileId = bawe.assemble(f, KeyFile.HEADER_LENGTH_PLUS_NAME_AMOUNT + nameLength);
        name = KeyFile.getKeyName(keyfileId);
      }
    } 
    catch (Exception e)
    {
      e.printStackTrace();
    }
    return name;
  }

  public String getKeyNameFromImg(File f) 
  {
    String name = "";
    try
    {
      BufferedImage img = ImageIO.read(f);
      // Check, if it is a PNG file.
      if (baie.containsHiddenData(f))
      {
        byte [] data = baie.assemble(img, KeyFile.HEADER_LENGTH_PLUS_NAME_AMOUNT + 4 + ByteArrayImgEncloser.ID.length);
        byte [] keyfileId =new byte [KeyFile.HEADER_LENGTH_PLUS_NAME_AMOUNT];
        System.arraycopy(data, 4 + ByteArrayImgEncloser.ID.length, keyfileId, 0, keyfileId.length);
        byte [] amountBytes = new byte [2]; 
        System.arraycopy(keyfileId, KeyFile.HEADER_LENGTH, amountBytes, 0, amountBytes.length);
        byte nameLength = HexTool.parseHex(new String(amountBytes));
        data = baie.assemble(img, KeyFile.HEADER_LENGTH_PLUS_NAME_AMOUNT + nameLength + 4 + ByteArrayImgEncloser.ID.length);
        keyfileId = new byte [KeyFile.HEADER_LENGTH_PLUS_NAME_AMOUNT + nameLength];
        System.arraycopy(data, 4 + ByteArrayImgEncloser.ID.length, keyfileId, 0, keyfileId.length);
        name = KeyFile.getKeyName(keyfileId);
      }
    } 
    catch (Exception e)
    {
      e.printStackTrace();
    }
    return name;
  }
}
