package de.das.encrypter.tools;

import de.das.encrypter.model.KeyFile;

/**
 * Tool for determining a quality to assess the keys.
 * 
 * @author Dipl.-Phys. Ing. Frank Keldenich
 */
public class KeyQualityAssessor
{
  public static final int QUALITY = 0;
  
  public static final int SCATTERING = 1;
  
  /**
   * Evaluates the quality of a key according to how equally distributed the 
   * occurring numerical values are as well as how equally distributed the 
   * distances of neighboring numerical values are.
   * 
   * @param key the key whose quality is to be evaluated.
   * 
   * @return an array of two numbers for evaluating the quality of the specified 
   * key.
   */
  public Double[] determineQuality (byte [] key)
  {
    double [] norm = new double [256];
    int [] counts = new int [256];
    int [] difCounts = new int [256];
    int byteValue;
    double minNorm = Double.MAX_VALUE;
    double maxNorm = Double.MIN_VALUE;
    
    byte [] amountBytes = new byte [2]; 
    System.arraycopy(key, KeyFile.HEADER_LENGTH, amountBytes, 0, amountBytes.length);
    int nn = HexTool.parseHex(new String(amountBytes));
    int startPoint = KeyFile.HEADER_LENGTH_PLUS_NAME_AMOUNT + nn;
    byte [] pureKey = new byte [key.length - (KeyFile.HEADER_LENGTH_PLUS_NAME_AMOUNT + nn + 4)];
    System.arraycopy(key, startPoint, pureKey, 0, pureKey.length);
    double q = 0;
    int prevByte = -1;
    for (byte b: pureKey)
    {
      byteValue = HexTool.byteToInt(b);
      counts[byteValue]++;
      if (prevByte >= 0)
      {
        difCounts[Math.abs(prevByte - byteValue)]++;
      }
      prevByte = byteValue;
    }
    
    double quality = 0;
    double max = 0;
    for (int v: counts)
    {
      if (v > max)
      {
        max = v;
      }
    }
    for (int i = 1; i < norm.length; i++)
    {
      norm[i] = counts[i] / max;
      if (norm[i] > maxNorm)
      {
        maxNorm = norm[i];
      }
      if (norm[i] < minNorm)
      {
        minNorm = norm[i];
      }
      quality = quality + norm[i];
    }
    
    Double [] res = new Double [2];
    res[QUALITY] = quality / 255;
    res[SCATTERING] = 1 - (maxNorm - minNorm);
    return res;
  }
}
